% Configure the compilation environment
% setenv('MW_MINGW64_LOC','D:\msys64\ucrt64');
% mex -setup C++

filePath = fullfile(pwd, 'htra_api_mat');  

% Check if the folder exists, if not, create the folder
if ~exist(filePath, 'dir')
    run('htra_api.m');
    filePath = fullfile(pwd, 'htra_api_mat');  
end

% Load htra_api.dll
if not(libisloaded('htra_api.dll'))   
    try
        % Try to load the dynamic link library and its header file
        loadlibrary('.\htra_api\htra_api.dll','.\htra_api\htra_api.h'); 
    catch ME
        % Catch any errors during the loading process, ME is an MException object containing error details
        fprintf('Failed to load htra_api.dll or htra_api.h: %s\n', ME.message);
        
        % End the program, please check whether htra_api.dll or htra_api.h files are in the htra_api folder at the same level as this .m file
        % If the files are not present, please copy the htra_api.dll or htra_api.h files from the HTRA_API folder in the accompanying materials (Windows\HTRA_API\x64) to the htra_api folder in the same directory as this .m file.
        fprintf('The program will end, please check whether the file path and file exist.\n');
        
        return;
        % End the current function or script execution
    end
end

% View all functions in the API
libfunctions('htra_api'); 

% Open the device
Device = libpointer; % Create a Device pointer
DevNum = 0;
Status = 0;

% Directly load BootProfile_TypeDef structure
load(fullfile(filePath, 'BootProfile.mat'));  
% Directly load BootInfo_TypeDef structure
load(fullfile(filePath, 'BootInfo.mat'));  

IS_USB = 1;  % Default to USB device, set to 0 for Ethernet device

if IS_USB == 1
    % Configure USB interface
    BootProfile.PhysicalInterface = 0;
else
    % Configure ETH interface
    BootProfile.PhysicalInterface = 3;
    BootProfile.ETH_IPVersion = 0;
    BootProfile.ETH_RemotePort = 5000;
    BootProfile.ETH_ReadTimeOut = 5000;
    BootProfile.ETH_IPAddress = [192, 168, 1, 100];
end

% Call Device_Open function
Status = calllib('htra_api', 'Device_Open', Device, DevNum, BootProfile, BootInfo);

if Status == 0
    get(BootProfile); % Print BootInfo values
    get(BootInfo); % Print BootInfo values    
    fprintf('Device successfully opened\n');
else
    % Device open failed, handle different error statuses
    switch Status
        case -1
            fprintf('Error - Check the power supply of the device, the connection of the data cable, and check that the driver is installed correctly\n');
        case -3
            fprintf('Error - RF calibration file is missing, please copy the RF calibration file to the CalFile folder\n');
        case -4
            fprintf('Error - IF calibration file is missing, please copy the IF calibration file to the CalFile folder\n');
        case -5
            fprintf('Error - Profile is missing, please copy the profile to the CalFile folder\n');
        case -6
            fprintf('Error - The device spec file is missing, please copy the device spec file to the CalFile folder\n');
        otherwise
            fprintf('Device turn on failed: Returns another error Status = %d', Status);
    end

    % Return error status value
    return ;
end

% Directly load IQS_Profile_TypeDef structure
load(fullfile(filePath, 'IQS_Profile.mat')); 

% Define IQS_ProfileIn pointer and IQS_ProfileOut pointer
IQS_ProfileIn = IQS_Profile;
IQS_ProfileOut = IQS_Profile;

% Call IQS_ProfileDeInit function
Status = calllib('htra_api', 'IQS_ProfileDeInit', Device, IQS_ProfileIn);
get(IQS_ProfileIn); % Print IQS_ProfileIn values

% Modify some parameters in IQS_ProfileIn structure
IQS_ProfileIn.CenterFreq_Hz = 1e9;   % Configure center frequency
IQS_ProfileIn.RefLevel_dBm = 0;      % Configure reference level
IQS_ProfileIn.DecimateFactor = 2;    % Configure decimation factor
IQS_ProfileIn.TriggerMode= 0;         % Configure trigger mode
IQS_ProfileIn.TriggerLength = 16384; % Configure the number of points per trigger

% Directly load IQS_StreamInfo_TypeDef structure
load(fullfile(filePath, 'IQS_StreamInfo.mat'));  

% Call IQS_Configuration function
Status = calllib('htra_api', 'IQS_Configuration', Device, IQS_ProfileIn, IQS_ProfileOut, IQS_StreamInfo);

% If configuration is successful (Status == 0)
if Status == 0
    get(IQS_ProfileOut); % Print IQS_ProfileOut values
    get(IQS_StreamInfo); % Print StreamInfo values
    fprintf('Configuration successful\n');
else
    % If an error occurs during the call, print the error message and close the device
    fprintf('IQS_Configuration Call error Status = %d\n', Status);
    
    % Call Device_Close function
    Device_Close(Device); 
    
    % Return 0, end function execution
    return;
end

% Directly load IQS_TriggerInfo_TypeDef structure 
load(fullfile(filePath, 'TriggerInfo.mat'));  
% Directly load MeasAuxInfo_TypeDef structure 
load(fullfile(filePath, 'MeasAuxInfo.mat')); 

IQ_data = int16(1:IQS_StreamInfo.PacketSamples * 2);

% Interleaved IQ time-domain data, single channel might be i8 i16 i32 format
AlternIQStream=libpointer('int16Ptr', IQ_data);

% Conversion factor from int type to absolute voltage (V)
ScaleToV = single(0);                                           
ScaleToV_p = libpointer('singlePtr', ScaleToV);

% Create I and Q arrays to store I and Q data separately
I_data = 1:IQS_StreamInfo.StreamSamples;
Q_data = 1:IQS_StreamInfo.StreamSamples;

% Get n times of IQ data
n = 10000;

% Decide the position of the IQS_BusTriggerStart and array space allocation size based on TriggerMode value
if strcmp(IQS_ProfileOut.TriggerMode, 'Adaptive')
    % When TriggerMode == Adaptive, execute IQS_BusTriggerStart first
    Status = calllib('htra_api', 'IQS_BusTriggerStart', Device);
    I_data = 1:IQS_StreamInfo.PacketSamples;
    Q_data = 1:IQS_StreamInfo.PacketSamples;
end

for j = 1:n
    if strcmp(IQS_ProfileOut.TriggerMode, 'FixedPoints')
        Status = calllib('htra_api', 'IQS_BusTriggerStart', Device);
    end
    for i = 1:IQS_StreamInfo.PacketCount

        if Status == 0
            Status = calllib('htra_api', 'IQS_GetIQStream', Device, AlternIQStream, ScaleToV_p, TriggerInfo, MeasAuxInfo);
            % Determine the number of valid points in the current packet, handle the last packet if the number of points is not a multiple
            if i == IQS_StreamInfo.PacketCount && ...
               mod(IQS_StreamInfo.StreamSamples, IQS_StreamInfo.PacketSamples) ~= 0 && ...
               strcmp(IQS_ProfileOut.TriggerMode, 'FixedPoints')
                Points = mod(IQS_StreamInfo.StreamSamples, IQS_StreamInfo.PacketSamples);
            else
                Points = IQS_StreamInfo.PacketSamples;
            end

            % Calculate the storage offset (MATLAB index starts from 1)
            offset = (i-1) * IQS_StreamInfo.PacketSamples;
            % Extract current data packet and convert
            currentIQ = double(AlternIQStream.Value(1:2*Points)) * ScaleToV_p.Value;

            % Separate I/Q data and store in arrays
            I_Data(offset+1 : offset+Points) = currentIQ(1:2:end);
            Q_Data(offset+1 : offset+Points) = currentIQ(2:2:end);
           
        else
            fprintf('IQS_GetIQStream Call error Status = %d\n', Status);
            return;
        end
    end
       
    % Dynamically plot accumulated data
    currentEnd = offset + Points;
    plot(1:currentEnd, I_Data(1:currentEnd), 1:currentEnd, Q_Data(1:currentEnd));

    % Dynamically adjust the axis
    currentY = [I_Data(1:currentEnd), Q_Data(1:currentEnd)];
    yMin = min(currentY);
    yMax = max(currentY);
    axis([0 currentEnd yMin-0.5 yMax+0.5]);

    pause(0.01);
    
end

% Call IQS_BusTriggerStop function
Status = calllib('htra_api', 'IQS_BusTriggerStop', Device);

% Call Device_Close function
Status = calllib('htra_api', 'Device_Close', Device);

clear all;
% Unload the library
unloadlibrary('htra_api');   
disp('Uninstall complete')

 
