% Configure compilation environment
% setenv('MW_MINGW64_LOC','D:\msys64\ucrt64');
% mex -setup C++

filePath = fullfile(pwd, 'htra_api_mat');  

% Check if the folder exists; if not, create it
if ~exist(filePath, 'dir')
    mkdir(filePath);  % Create the target folder
    run('htra_api.m');
    filePath = fullfile(pwd, 'htra_api_mat');  
end

% Load htra_api.dll
if not(libisloaded('htra_api.dll'))   
    loadlibrary('.\htra_api\htra_api.dll','.\htra_api\htra_api.h'); % Make sure the paths to .dll and .h files are correct
end

libfunctions('htra_api'); % View all functions in the API

% Open the device
Device = libpointer; % Create Device pointer
DevNum = 0;
Status = 0;

load(fullfile(filePath, 'BootProfile.mat'));  % Directly load the BootProfile_TypeDef structure
load(fullfile(filePath, 'BootInfo.mat'));  % Directly load the BootInfo_TypeDef structure

IS_USB = 1;  % Default to USB device; set to 0 if using an Ethernet device

if IS_USB == 1
    % Configure USB interface
    BootProfile.PhysicalInterface = 0;
else
    % Configure ETH interface
    BootProfile.PhysicalInterface = 3;
    BootProfile.ETH_IPVersion = 0;
    BootProfile.ETH_RemotePort = 5000;
    BootProfile.ETH_ReadTimeOut = 5000;
    BootProfile.ETH_IPAddress = [192, 168, 1, 100];
end

% Call Device_Open function
Status = calllib('htra_api', 'Device_Open', Device, DevNum, BootProfile, BootInfo);

if Status == 0
    get(BootProfile); % Print values of BootProfile
    get(BootInfo);    % Print values of BootInfo    
    fprintf('Device successfully\n');
else
    % Device opening failed, handle different error statuses
    switch Status
        case -1
            fprintf('Error - Check the power supply of the device, the connection of the data cable, and check that the driver is installed correctly\n');
        case -3
            fprintf('Error - RF calibration file is missing, please copy the RF calibration file to the CalFile folder\n');
        case -4
            fprintf('Error - IF calibration file is missing, please copy the IF calibration file to the CalFile folder\n');
        case -5
            fprintf('Error - Profile is missing, please copy the profile to the CalFile folder\n');
        case -6
            fprintf('Error - The device spec file is missing, please copy the device spec file to the CalFile folder\n');
        otherwise
            fprintf('Device turn on failed: Returns another error Status = %d', Status);
    end

    % Return error status value
    return ;
end

load(fullfile(filePath, 'SWP_Profile.mat'));  % Directly load the SWP_Profile_TypeDef structure

% Define SWP_ProfileIn pointer and SWP_ProfileOut pointer
SWP_ProfileIn = SWP_Profile;
SWP_ProfileOut = SWP_Profile;

% Call SWP_ProfileDeInit function
Status = calllib('htra_api', 'SWP_ProfileDeInit', Device, SWP_ProfileIn);
get(SWP_ProfileIn); % Print values of SWP_ProfileIn

% Modify some parameters of the SWP_ProfileIn structure
SWP_ProfileIn.StartFreq_Hz = 9e3;  % Configure start frequency
SWP_ProfileIn.StopFreq_Hz = 3e9;   % Configure stop frequency
SWP_ProfileIn.RefLevel_dBm = 0;    % Configure reference level
SWP_ProfileIn.RBWMode=0;           % Configure RBWMode
SWP_ProfileIn.RBW_Hz = 300e3;      % Configure RBW

load(fullfile(filePath, 'TraceInfo.mat'));  % Directly load the SWP_TraceInfo_TypeDef structure

% Call SWP_Configuration function
Status = calllib('htra_api', 'SWP_Configuration', Device, SWP_ProfileIn,SWP_ProfileOut,TraceInfo);

% If configuration is successfully sent (Status == 0)
if Status == 0
    get(SWP_ProfileOut); % Print values of SWP_ProfileOut
    get(TraceInfo);      % Print values of TraceInfo
    fprintf('Configuration successfully\n');
else
    % If the call fails, print error info and close the device
    fprintf('SWP_Configuration Call error Status = %d\n', Status);
    Device_Close(Device); 
    
    % Return 0, end function execution
    return;
end

% Create Frequency to store SWP frequency data, and PowerSpec_dBm to store amplitude data
Frequency = double(1:TraceInfo.FullsweepTracePoints);
PowerSpec_dBm = single(1:TraceInfo.FullsweepTracePoints);

% Convert C data types to MATLAB types
Frequency_p = libpointer('doublePtr',Frequency);
PowerSpec_dBm_p = libpointer('singlePtr',PowerSpec_dBm);

% The full sweep of this device is composed of multiple frames, this parameter returns the sequence number of the current frame
HopIndex = int32(0);
% Indicates the index of the current frame among all frames
FrameIndex = int32(0);
% Create pointer HopIndex_p
HopIndex_p = libpointer('int32Ptr', HopIndex);
% Create pointer FrameIndex_p
FrameIndex_p = libpointer('int32Ptr', FrameIndex);

load(fullfile(filePath, 'MeasAuxInfo.mat'));  % Directly load the MeasAuxInfo_TypeDef structure

% Get SWP data n times
n=10000; 
for t=1:n
    % TraceInfo.TotalHops indicates the number of frames under current configuration,
    % so call SWP_GetPartialSweep TraceInfo.TotalHops times to get the full trace
    for i=1:TraceInfo.TotalHops 
        Status = calllib('htra_api', 'SWP_GetPartialSweep', Device, Frequency_p + (i-1) * TraceInfo.PartialsweepTracePoints, PowerSpec_dBm_p + (i-1) * TraceInfo.PartialsweepTracePoints,HopIndex_p,FrameIndex_p,MeasAuxInfo);     
        if Status ~= 0
            % If the call fails, print error info and close the device
            fprintf('SWP_Configuration Call error Status = %d\n', Status);
            % Return 0, end function execution
            return;
        end
    end
    
    nPeaks = 6;  % Assume a maximum of 6 peaks
    [Power_Max, Freq_Max] = deal([]);  % Initialize
    Power_temp = PowerSpec_dBm_p.Value;

    % Plot the SWP spectrum
    plot(Frequency_p.Value,PowerSpec_dBm_p.Value);
    ylim([-130, SWP_ProfileOut.RefLevel_dBm]);
    xlim([min(Frequency_p.Value), max(Frequency_p.Value)]);
    hold on;
    
    for i = 1:nPeaks
        [maxValue, maxIndex] = max(Power_temp);  % Find the max value and its index
        Power_Max(i) = maxValue;                 % Store the max value
        Freq_Max(i)=Frequency_p.Value(maxIndex);
        Power_temp(maxIndex) = -Inf;             % Set this value to -Inf to find next max

        bar(Frequency_p.Value(maxIndex),maxValue,'r','LineStyle', ':');
        text(Frequency_p.Value(maxIndex), SWP_ProfileOut.RefLevel_dBm, char('a' + i - 1), 'HorizontalAlignment', 'center', 'VerticalAlignment', 'bottom');
        text(0.05, 1-0.05*i, [char('a' + i - 1)  ' Frequency ' num2str(Frequency_p.Value(maxIndex)),' Power ' num2str(maxValue)], 'Units', 'normalized');
    end
    
    xlabel('Frequency');
    ylabel('Power');
    pause(0.01); 
    clf;
end

% Call Device_Close function
Status = calllib('htra_api', 'Device_Close', Device);

clear all;

% Unload the library
unloadlibrary('htra_api');   
disp('Uninstall complete');
