using System.Runtime.InteropServices;
using System;

using uint8_t = System.Byte;
using uint16_t = System.UInt16;
using uint32_t = System.UInt32;
using uint64_t = System.UInt64;
using int8_t = System.SByte;
using int16_t = System.Int16;
using int32_t = System.Int32;
using int64_t = System.Int64;

public class HtraApi
{

    /**
    * @brief    This file defines variables and functions related to the API. Note to use UTF-8 with BOM for editing.
*/

    public const int Major_Version = 0;
    public const int Minor_Version = 55;
    public const int Increamental_Version = 63;

    /* The key user data types and user functions are described below

    User data types are the core content for users to configure devices and return information from the devices. Users utilize these user data types and user functions to implement the use of the API. The data types used in the user program are as follows:

    Device-related data types
    BootProfile_TypeDef:The configuration information required for device startup.
    BootInfo_TypeDef:Device startup process feedback information.

    Device-related main functions
    Device_Open(..): Opens the device. The device must be opened first (allocating software and hardware resources, such as memory and USB ports) before configuration and calling.
    Device_Close(..): Closes the device. After the device is finished using, it needs to be closed to allow the operating system to recover software and hardware resources.

    Standard Spectrum Mode (SWP) related data types
    SWP_Profile_TypeDef: Configuration information for the SWP mode.
    SWP_TraceInfo_TypeDef: Configuration feedback information (trace information) for the SWP mode
    Freq_Hz[], PowerSpec_dBm[], HopIndex, FrameIndex, MeasAuxInfo:Measurement data and additional information obtained by SWP, in non-encapsulated form.
    SWPSpectrum_TypeDef: Measurement data and additional information obtained by SWP, in encapsulated form.

    Standard spectrum mode (SWP) related main functions include:
    SWP_ProfileDeInit(..,SWP_Profile): Initializes the SWP_Profile configuration information to default values.
    SWP_Configuration(..,SWP_Profile, SWP_TraceInfo): Configures the device in SWP mode with the configuration information SWP_Profile and obtains the configuration feedback information SWP_TraceInfo.
    SWP_GetPartialSweep(.., Freq_Hz[], PowerSpec_dBm[], HopIndex, FrameIndex, MeasAuxInfo): Retrieves measurement data in SWP mode, partial traces, using standard data structures, including main information.
    SWP_GetPartialSweep_PM1(.., SWPTrace_TypeDef): Retrieves measurement data in SWP mode, partial traces, using an encapsulated data structure, including all information.

    IQ stream mode (IQS) related data types include:
    IQS_Profile_TypeDef: Configuration information for the IQS mode.
    IQS_StreamInfo_TypeDef: Configuration feedback information for the IQS mode.
    IQStream_TypeDef: Measurement data obtained in the IQS mode.

    IQ stream mode (IQS) related main functions include:
    IQS_ProfileDeInit(..)
    IQS_Configuration(..)
    IQS_BusTriggerStart(..)
    IQS_BusTriggerStop(..)
    IQS_GetIQStream(..,AlternIQStream, ScaleToV, TriggerInfo, MeasAuxInfo)
    IQS_GetIQStream_PM1(..,IQStream_TypeDef)

    Detection mode (DET) related data types include:
    DET_Profile_TypeDef: Configuration information for the DET mode.
    DET_StreamInfo_TypeDef:
    DETStream_TypeDef

    Detection mode (DET) related main functions include:
    DET_ProfileDeInit
    DET_Configuration
    DET_BusTriggerStart
    DET_BusTriggerStop
    DET_GetPowerStream(.., PowerStream[], ScaleToV, TriggerInfo, MeasAuxInfo)

    Real-time spectrum mode (RTA) related data types include:
    RTA_Profile_TypeDef: Configuration information for the RTA mode.
    RTA_FrameInfo_TypeDef: Feedback information from the configuration process in RTA mode.

    SpectrumFrames[], SpectrumBitMap[], TriggerInfo, PlotInfo, MeasAuxInfo unencapsulated measurement data in RTA mode, matching the RTA_GetRealTimeSpectrum function.
    RTAStream_TypeDef:unencapsulated measurement data in RTA mode, matching the RTA_GetRealTimeSpectrum_PM1 functiom.

    The main function of real-time spectral mode (RTA)
    RTA_ProfileDeInit
    RTA_Configuration
    RTA_BusTriggerStart
    RTA_BusTriggerStop
    RTA_GetRealTimeSpectrum(..,SpectrumFrames[], SpectrumBitMap[], TriggerInfo, PlotInfo, MeasAuxInfo)
    RTA_GetRealTimeSpectrum_PM1(RTAStream_TypeDef)

    */

    /* Return value definition  Return Value for public API functions-------------------------------------------------------------*/

    /* Description of the return value
    1) 0 is the default normal value. Any value other than 0 indicates a special condition.
    2) The prefix APIRETVAL_ indicates that this return value is an API return value.
    3) The second modifier ERROR_ indicates that the returned condition is an error. In this case, it is not recommended to directly perform subsequent operations. You need to try specific recovery steps, such as re-running DeviceOpen.
    4) The second modifier WARNING_ indicates that the return condition is warning, and subsequent API calls can be continued. However, the test data obtained by the current API may be inaccurate, for example, local vibration loss lock may cause power reading miscalibration.
    */

    public const int APIRETVAL_NoError = 0;// No Error

    /* Errors and warnings during device open process */
    public const int APIRETVAL_ERROR_BusOpenFailed = -1;                // Bus opening error, only in DeviceOpen procedure.
    public const int APIRETVAL_ERROR_RFACalFileIsMissing = -3;          // RF amplitude calibration file is lost. This file is produced only in DeviceOpen. If the file is lost, the amplitude calibration will be out of alignment or cannot run.#define	APIRETVAL_ERROR_RFACalFileIsMissing
    public const int APIRETVAL_ERROR_IFACalFileIsMissing = -4;          // If the amplitude calibration file is lost, produced only in the DeviceOpen process.If the file is lost, the amplitude calibration will be out of alignment or cannot run.
    public const int APIRETVAL_ERROR_DeviceConfigFileIsMissing = -5;    // The device configuration file is lost and is produced only in the DeviceOpen process. The configuration is not optimized.
    public const int APIRETVAL_ERROR_DeviceSpecFileIsMissing = -6;      // The device specification file is lost and is produced only in the DeviceOpen process. The parameter range is incorrect.
    public const int APIRETVAL_ERROR_UpdateStrategyFailed = -7;         // The process of delivering the configuration policy to the device fails. It is produced only in the process of DeviceOpen.

    /* Errors and warnings during bus invocation */
    public const int APIRETVAL_ERROR_BusError = -8;			// Bus transmission error.
    public const int APIRETVAL_ERROR_BusDataError = -9;		//  Bus data error means that the overall format of the data packet is correct, but the data content is wrong, which may be due to configuration conflict or physical interference during transmission
    public const int APIRETVAL_WARNING_BusTimeOut = -10;	// The bus data return time out, either because the trigger did not come, or because there was a communication error.
    public const int APIRETVAL_ERROR_BusDownLoad = -11;		// The configuration is incorrectly delivered through the bus.

    /* Errors and warnings in data packets about data content and device status */
    public const int APIRETVAL_WARNING_IFOverflow = -12;                    // If the intermediate frequency is saturated, it is recommended to lower the reference level(RefLevel_dBm) until this prompt no longer appears.
    public const int APIRETVAL_WARNING_ReconfigurationIsRecommended = -14;  // Since the temperature has changed significantly since the last configuration, it is recommended to re-invoke configuration functions (such as SWP_Configuration) for best performance.
    public const int APIRETVAL_WARNING_ClockUnlocked_SYSCLK = -15;          // Hardware status warning:the system clock is out of lock.
    public const int APIRETVAL_WARNING_ClockUnlocked_ADCCLK = -16;          // Hardware status warning:ADC clock is out of lock.
    public const int APIRETVAL_WARNING_ClockUnlocked_RXIFLO = -17;          // Hardware status warning:the received IF local oscillator loss lock.
    public const int APIRETVAL_WARNING_ClockUnlocked_RXRFLO = -18;          // Hardware status warning:the received RF local oscillator lost lock.
    public const int APIRETVAL_WARNING_ADCConfigError = -19;                // Hardware status warning:ADC configuration error.  

    /* An error occurred during firmware update */
    public const int APIRETVAL_ERRPR_OpenfileFiled = -20;               // Failed to open the update file.
    public const int APIRETVAL_ERROR_ExecuteUpdateError = -21;          // Failed to enter the update program.
    public const int APIRETVAL_ERROR_FirmwareEraseError = -22;          // Failed to erase firmware flash.
    public const int APIRETVAL_ERROR_FirmwareDownloadError = -23;       // Failed to download the firmware flash.
    public const int APIRETVAL_ERROR_UpdateExitError = -24;             // Failed to exit the firmware update program.
    public const int APIRETVAL_ERROR_DisallowUpgrade = -301;		    // Upgrade not allowed
    public const int APIRETVAL_ERROR_PowerRestart = -302;		        // Power restart failed


    /* Error state when setting standby mode*/
    public const int APIRETVAL_ERROR_SysPowerMode_ParamOutRange = -25;// Failed to exit the firmware update program
    public const int APIRETVAL_ERROR_SysPowerMode_UpdateFailed = -26; // Failed to exit the firmware update program.

    public const int APIRETVAL_WARNING_VNACalKitFileIsMissing = -27;	// Failed to load VNA calibration component data file.
    public const int APIRETVAL_WARNING_VNACalResultFileIsMissing = -28; // Failed to load the VNA calibration file.

    public const int APIRETVAL_WARNING_TxLevelCalFileIsMissing = -29;// Failed to load the transmit power calibration file.
    public const int APIRETVAL_WARNING_TxLevelExceededRefLevel = -30;// Failed to load the transmit power calibration file.

    /* Calibration file loading warning */
    public const int APIRETVAL_WARNING_DefaultRFACalFileIsLoaded = -33;// The RF amplitude calibration file of the device is lost. The system loads the default calibration file.
    public const int APIRETVAL_WARNING_DefaultIFACalFileIsLoaded = -34;// The IF amplitude calibration file of the device is lost. The system loads the default calibration file.

    /* Device state warning*/
    public const int APIRETVAL_WARNING_ClockRelocked_SYSCLK = -36;// Hardware state warning, system clock re-lock
    public const int APIRETVAL_WARNING_ClockRelocked_ADCCLK = -37;// Hardware state warning, ADC clock re-lock
    public const int APIRETVAL_WARNING_ClockRelocked_RXIFLO = -38;// Hardware state warning, IF LO re-lock
    public const int APIRETVAL_WARNING_ClockRelocked_RXRFLO = -39;// Hardware state warning, RF LO re-lock

    public const int APIRETVAL_WARNING_DefaultVNACalKitFileIsLoaded = -40;      // The vector network analyzer calibration file of the device is lost, and the system loads the calibration file by default.
    public const int APIRETVAL_WARNING_DefaultVNACalResultFileIsLoaded = -41;   // The calibration result file of the vector network analyzer of the device is lost. The system loads the calibration result file by default.
    public const int APIRETVAL_WARNING_DefaultTxLevelCalFileIsLoaded = -42;     // The transmitted power calibration file of the individual device is lost. The system loads the transmitted power calibration file by default.

    public const int APIRETVAL_ERROR_IQCalFileIsMissing = -43;          // IQ calibration file loss, only produced in the DeviceOpen process, the file loss will cause IQ calibration misalignment or unable to run.
    public const int APIRETVAL_WARNING_DefaultIQCalFileIsLoaded = -44;  // The IQ calibration file of the individual device is lost, and the system loads the default calibration.

    public const int APIRETVAL_ERROR_QueryResultsIncorrect = -45;       // The query result does not match the actual delivered parameters.
    public const int APIRETVAL_ERROR_ADCState = -46;                    // The ADC status is incorrect.
    public const int APIRETVAL_ERROR_Options_Missing = -47;             // The hardware status is incorrect. The required options do not exist. Please check whether the MUXIO connection is normal.
    public const int APIRETVAL_ERROR_CableLossFileIsMissing = -48;      // Loss of line loss files, produced only during DeviceOpen process.

    /* Abnormal call and debugging errors and warnings */
    public const int APIRETVAL_ERROR_ParamOutRange = -98;            // Parameter out of bounds, not called properly.
    public const int APIRETVAL_ERROR_FunctionInternalError = -99;    // Function internal error, abnormal call.
    public const int APIRETVAL_WARNING_No_VSG_Function_Option = -100;// The IQ calibration file of the individual device is lost, and the system loads the default calibration file.

    /* Abnormal call resampling function errors */
    public const int APIRETVAL_ERROR_DecimateFactorOutRange = -201;// Decimate factor setting exceeds the limit,abnormal call(The decimate factor are set in the range 1 ~ 2^16)
    public const int APIRETVAL_ERROR_IQSDataFormatError = -202;    // IQS Data tpye error,abnormal call(The IQ date type is currently only int8、int16、int32 and float)
    public const int APIRETVAL_WARNING_CarrierLoss = -203;		   // No signals with power exceeding the carrier judgment threshold were detected during phase noise measurements.
    public const int APIRETVAL_WARNING_MeasUpdate = -204;          // During phase noise measurement, the DUT state changes, resulting in automatic update of the measurement state and the need to re-acquire the PartialUpdateCounts sub-data.	


    public const int APIRETVAL_ERROR_ETHTimeOut = 10060;           // The connection attempt fails due to no correct reply from the connecting party after some time or no reaction from the connected host. 
    public const int APIRETVAL_ERROR_ETHDisconnected = 10054;      // The device disconnects from the network, and the remote host forces the closure of an existing connection.
    public const int APIRETVAL_ERROR_ETHDataError = 10062;         // The device is not retrieving data properly.

    public const int APIRETVAL_ERROR_NoDeviceExist = 10068;        // No device exists
    public const int APIRETVAL_ERROR_DeviceNotFound = 10069;       // The device whose IP address you want to change cannot be found.
    public const int APIRETVAL_ERROR_DeviceIPRepeatConfig = 10070; // The ip address range does not match
    public const int APIRETVAL_ERROR_DeviceIPExceed = 10071;       // The ip address range does not match
    public const int APIRETVAL_ERROR_DeviceIPSetError = 10072;     // When the device is connected to a router, the IP address set must meet routing requirements
    public const int APIRETVAL_ERROR_DeviceIPExist = 10073;        // The IP address to be changed exists in this network segment
    public const int APIRETVAL_ERROR_ConfigDeviceIPFailed = 10074; // Failed to configure the IP address
    public const int APIRETVAL_ERROR_DeviceInfoError = 10079;      // Incorrect device information is obtained


    /* Data type obtained in MPS mode */
    public const int MPS_SWP = 0x00;// When Analysis returns 0, it is SWP mode.        
    public const int MPS_IQS = 0x01;// When Analysis returns 1, it is IQS mode. 
    public const int MPS_DET = 0x02;// When Analysis returns 2, it is DET mode.  
    public const int MPS_RTA = 0x03;// When Analysis returns 3, it is RTA mode.

    /*---------------------------------------------------
     Enumeration required by the device start-stop structure
    -----------------------------------------------------*/

    /*Equipment power supply mode BootProfile.DevicePowerSupply*/
    public enum DevicePowerSupply_TypeDef
    {
        USBPortAndPowerPort = 0x00,    // Use USB data port and power port for power supply.
        USBPortOnly = 0x01,            // Only USB data ports are used.
        Others = 0x02                  // Other modes: If the ETH bus is used, select this option.
    };

    /*Physical bus type BootProfile.PhysicalInterface*/
    public enum PhysicalInterface_TypeDef
    {
        USB = 0x00,    // Uses USB as the physical interface, applicable to products with USB interfaces, such as SAE and SAM.
        QSFP = 0x02,   // Use 40Gbps QSFP+ as a physical interface.
        ETH = 0x03,    // Uses 100M/1000M Ethernet as the physical interface and applies to Ethernet interface products such as NXEs and NXMS.
        HLVDS = 0x04,  //  Only for debugging, do not use.
        VIRTUAL = 0x07 //  Use a virtual bus, that is, no physical bus, for simulation and debugging.
    };

    /*IP address version  BootProfile.IPVersion*/
    public enum IPVersion_TypeDef
    {
        IPv4 = 0x00, // Use an IPv4 address.
        IPv6 = 0x01  // Use an IPv6 address.
    };

    /*Power consumption control*/
    public enum SysPowerState_TypeDef
    {
        PowerON = 0x00,    // All workspaces in the system are powered on.
        RFPowerOFF = 0x01, // The RF is powered off and cannot be woken up quickly.
        RFStandby = 0x02   // The RF is in standby state and can wake up quickly.
    };

    /*Firmware type*/
    public enum Firmware_TypeDef
    {
        MCU = 0x00,     // SCM
        FPGA = 0x01,    // FPGA
        GNSS = 0x02     // GNSS
    };

    /*---------------------------------------------------
    Enumeration required by the user structure, with the adaptation working mode in parentheses ().
    all:Including the full mode 
    SWP:Scan mode 
    IQS:Time domain flow mode
    RTA:Real-time spectrum analysis mode
    DET:Detection mode 
    DSP:Digital signal processing function 
    VNA:Vector network analysis mode .
    SIG:Auxiliary signal generator mode .
    -----------------------------------------------------*/

    /*Rf input port(all)*/
    public enum RxPort_TypeDef
    {
        ExternalPort = 0x00,    // External port.
        InternalPort = 0x01,    // Internal port
        ANT_Port = 0x02,        // only for TRx series
        TR_Port = 0x03,         // only for TRx series
        SWR_Port = 0x04,        // only for TRx series
        INT_Port = 0x05         // only for TRx series
    };

    /*Gain mode(all)*/
    public enum GainStrategy_TypeDef
    {
        LowNoisePreferred = 0x00,       // Low noise.
        HighLinearityPreferred = 0x01   // High linearity.
    };

    /*Trigger output type(all)*/
    public enum TriggerEdge_TypeDef
    {
        RisingEdge = 0x00,      // Rising edge triggers.
        FallingEdge = 0x01,     // Falling edge triggers.
        DoubleEdge = 0x02       // Two-sided edge triggers.
    };

    /*Trigger output type(all)*/
    public enum TriggerOutMode_TypeDef
    {
        None = 0x00,        // No trigger output.
        PerHop = 0x01,      //Output with each completion of frequency hopping.
        PerSweep = 0x02,    // Output with the completion of each scan.
        PerProfile = 0x03   // Output with each configuration switch.
    };

    /*Trigger the output signal polarity(all)*/
    public enum TriggerOutPulsePolarity_TypeDef
    {
        Positive = 0x00, //  positive type pulse.
        Negative = 0x01  // negative type pulse.
    };

    /*Preselected amplifier(all)*/
    public enum PreamplifierState_TypeDef
    {
        AutoOn = 0x00,      // Automatically enables the preamplifier.
        ForcedOff = 0x01    //  Force to keep the preamplifier off.
    };

    /*System reference clock(all)*/
    public enum ReferenceClockSource_TypeDef
    {
        ReferenceClockSource_Internal = 0x00,           // Internal clock source (Default: 10MHz).
        ReferenceClockSource_External = 0x01,           // External clock source (Default: 10MHz). When the system detects that the external clock source cannot be locked, the system automatically switches to the internal reference.
        ReferenceClockSource_Internal_Premium = 0x02,   // Internal Clock Source - High quality, choose DOCXO or OCXO.
        ReferenceClockSource_External_Forced = 0x03     // External clock source, and regardless of locking, will not switch to internal reference even if the lock is lost.
    };

    /*System clock(all)*/
    public enum SystemClockSource_TypeDef
    {
        SystemClockSource_Internal = 0x00, // Internal system clock source.
        SystemClockSource_External = 0x01  // External system clock source.
    };

    /*Frequency allocation mode(SWP)*/
    public enum SWP_FreqAssignment_TypeDef
    {
        StartStop = 0x00, // Set the frequency range by the start and the stop frequency.
        CenterSpan = 0x01 // Set the frequency range by center frequency and span.
    };

    /*Trace updating mode(SWP)*/
    public enum SWP_TraceType_TypeDef
    {
        ClearWrite = 0x00,      // Output the normal trace.
        MaxHold = 0x01,         // Output the trace maintained by the maximum value.
        MinHold = 0x02,         // Output the trace maintained by the minimum value.
        ClearWriteWithIQ = 0x03 // Output time domain data and frequency domain data of current frequency point at the same time.
    };

    /*RBW update mode(SWP)*/
    public enum RBWMode_TypeDef
    {
        RBW_Manual = 0x00,              // Enter RBW manually.
        RBW_Auto = 0x01,                // Update RBW automatically with SPAN,RBW = SPAN / 2000
        RBW_OneThousandthSpan = 0x02,   // Mandatory  RBW = 0.001 * SPAN
        RBW_OnePercentSpan = 0x03       // Mandatory  RBW = 0.01 * SPAN
    };

    /*VBW update mode(SWP)*/
    public enum VBWMode_TypeDef
    {
        VBW_Manual = 0x00,          // Enter VBW manually.
        VBW_EqualToRBW = 0x01,      // Mandatory VBW = RBW.
        VBW_TenPercentRBW = 0x02,   // Mandatory VBW = 0.1 * RBW.
        VBW_OnePercentRBW = 0x03,   // Mandatory VBW = 0.01 * RBW
        VBW_TenTimesRBW = 0x04      // Mandatory  VBW = 10 * RBW,Fully bypass VBW filter
    };

    /*SwpTime_Mode*/
    public enum SweepTimeMode_TypeDef
    {
        SWTMode_minSWT = 0x00,      // Sweep with minimum sweep time.
        SWTMode_minSWTx2 = 0x01,  // Sweep with approximately 2 times of the minimum sweep time.
        SWTMode_minSWTx4 = 0x02,  // Sweep with approximately 4 times of the minimum sweep time.
        SWTMode_minSWTx10 = 0x03, // Sweep with approximately 10 times of the minimum sweep time.
        SWTMode_minSWTx20 = 0x04, // Sweep with approximately 20 times of the minimum sweep time.
        SWTMode_minSWTx50 = 0x05, // Sweep with approximately 50 times of the minimum sweep time.
        SWTMode_minSWTxN = 0x06,  // Sweep with approximately N times of the minimum sweep time，N=SweepTimeMultiple.
        SWTMode_Manual = 0x07,    // Sweep at approximately the specified sweep time, which is equal to SweepTime.
        SWTMode_minSMPxN = 0x08   // A single frequency point is sampled with approximately N times the shortest sampling time, N=SampleTimeMultiple
    };

    /*Scan mode trigger source and trigger mode (SWP)*/
    public enum SWP_TriggerSource_TypeDef
    {
        InternalFreeRun = 0x00,         // Internal trigger free running
        ExternalPerHop = 0x01,          // External trigger, each trigger jumps to a frequency point
        ExternalPerSweep = 0x02,        // External trigger, each trigger refreshes a trace line
        ExternalPerProfile = 0x03       // External trigger, each trigger applies a new configuration
    };

    /*Spurious rejection type(SWP)*/
    public enum SpurRejection_TypeDef
    {
        Bypass = 0x00,      // No spurious rejection
        Standard = 0x01,    // Standard spurious rejection.
        Enhanced = 0x02     // Enhanced spurious rejection.
    };

    /*Trace point approximation method(SWP)*/
    public enum TracePointsStrategy_TypeDef
    {
        SweepSpeedPreferred = 0x00,         // Give priority to the fastest scanning speed, and then try to get as close as possible to the set target trace points.
        PointsAccuracyPreferred = 0x01,     // The priority is to ensure that the actual trace points are close to the set target trace points.
        BinSizeAssined = 0x02               // The priority is to ensure that the trace is generated at the set frequency interval.
    };

    /*Trace aligment(SWP)*/
    public enum TraceAlign_TypeDef
    {
        NativeAlign = 0x00,  //Natural alignment.
        AlignToStart = 0x01, //Align to start frequency.
        AlignToCenter = 0x02 //Align to center frequency.
    };

    /*FFT Execution Strategy(SWP)*/
    public enum FFTExecutionStrategy_TypeDef
    {
        Auto = 0x00,                    // the CPU or FPGA is automatically selected for FFT calculation
        Auto_CPUPreferred = 0x01,       // the CPU or FPGA is automatically selected for FFT calculation, CPU is preferred.
        Auto_FPGAPreferred = 0x02,      // the CPU or FPGA is automatically selected for FFT calculation, FPGA is preferred.
        CPUOnly_LowResOcc = 0x03,       // Forced use of CPU calculation, low resource occupation, maximum FFT points 256K.
        CPUOnly_MediumResOcc = 0x04,    // Forced use of CPU calculation, medium resource occupation, maximum FFT points 1M.
        CPUOnly_HighResOcc = 0x05,      // Forced use of CPU calculation, high resource occupation, maximum FFT points 4M.
        FPGAOnly = 0x06                 // Forced use of FPGA calculation.
    };

    /*DSP calculating platform (SWP)*/
    public enum DSPPlatform_Typedef
    {
        CPU_DSP = 0x00,     // CPU
        FPGA_DSP = 0x01     // FPGA
    };

    /*Multi - frame detection in frequency point(SWP\RTA)*/
    public enum Detector_TypeDef
    {
        Detector_Sample = 0x00,     // No inter-frame detection
        Detector_PosPeak = 0x01,    // MaxHold Frame detection is carried between power spectrum of each frequency point
        Detector_Average = 0x02,    // Average Frame detection is carried between power spectrum of each frequency point
        Detector_NegPeak = 0x03,    // MinHold Frame detection is carried between power spectrum of each frequency point
        Detector_MaxPower = 0x04,   // Each frequency point is sampled for a long time before FFT and frame spectrum with highest power is selected for FFT to capture instantaneous signals such as pulse. (SWP mode only)
        Detector_RawFrames = 0x05,  // Multiple sampling, multiple FFT analyses for each frequency, and frame-by-frame output power spectrum (SWP mode only)
        Detector_RMS = 0x06,        // RMS Frame detection is carried between power spectrum of each frequency point
        Detector_AutoPeak = 0x07    // Aoto Peak Frame detection is carried between power spectrum of each frequency point
    };

    /*Window type(SWP\RTA\DSP)*/
    public enum Window_TypeDef
    {
        FlatTop = 0x00,             // Flat top window.
        Blackman_Nuttall = 0x01,    // Nuttall window.
        LowSideLobe = 0x02		    // LowSideLobe window        
    };

    /* Trace detector mode (SWP\RTA\DSP)*/
    public enum TraceDetectMode_TypeDef
    {
        TraceDetectMode_Auto = 0x00,  // automatically selection of a detection.
        TraceDetectMode_Manual = 0x01 // specification of a detection.
    };

    /*Detection mode(SWP\RTA\DSP)*/
    public enum TraceDetector_TypeDef
    {
        TraceDetector_AutoSample = 0x00,    // auto sample detection.
        TraceDetector_Sample = 0x01,        // sample detection.
        TraceDetector_PosPeak = 0x02,       // positive peak detection.
        TraceDetector_NegPeak = 0x03,       // negative peak detection.
        TraceDetector_RMS = 0x04,           // RMS detection.
        TraceDetector_Bypass = 0x05,        // no detection.
        TraceDetector_AutoPeak = 0x06 	    // auto peak detection.
    };

    /*Fixed frequency point mode (IQS\RTA\DET) trigger source*/
    public enum IQS_TriggerSource_TypeDef
    {
        FreeRun = 0x00,                // Free running
        External = 0x01,               // External trigger. Triggered by a physical signal connected to a trigger input port outside the device.
        Bus = 0x02,                    // The bus is triggered. Triggered by a function (instruction).
        Level = 0x03,                  // Level trigger. The device detects the input signal according to the set level threshold, and triggers automatically when the input exceeds the threshold.
        Timer = 0x04,                  // The timer is triggered. Use the device internal timer to automatically trigger the set time period.
        TxSweep = 0x05,                // Output trigger of signal generator scan; When this trigger source is selected, the acquisition process will be triggered by the output trigger signal scanned by the signal source.
        MultiDevSyncByExt = 0x06,      // When the external trigger signal arrives, multiple machines perform synchronous trigger behavior.
        MultiDevSyncByGNSS1PPS = 0x07, // When GNSS-1PPS arrives, multiple machines trigger synchronously.
        SpectrumMask = 0x08,           // Spectrum template trigger, available only in RTA mode. Not available in this stage.
        GNSS1PPS = 0x09				   // Use 1PPS provided by GNSS in the system to trigger.
    };

    public enum DET_TriggerSource_TypeDef
    {
        FreeRun = 0x00,                // Free running
        External = 0x01,               // External trigger. Triggered by a physical signal connected to a trigger input port outside the device.
        Bus = 0x02,                    // The bus is triggered. Triggered by a function (instruction).
        Level = 0x03,                  // Level trigger. The device detects the input signal according to the set level threshold, and triggers automatically when the input exceeds the threshold.
        Timer = 0x04,                  // The timer is triggered. Use the device internal timer to automatically trigger the set time period.
        TxSweep = 0x05,                // Output trigger of signal generator scan; When this trigger source is selected, the acquisition process will be triggered by the output trigger signal scanned by the signal source.
        MultiDevSyncByExt = 0x06,      // When the external trigger signal arrives, multiple machines perform synchronous trigger behavior.
        MultiDevSyncByGNSS1PPS = 0x07, // When GNSS-1PPS arrives, multiple machines trigger synchronously.
        SpectrumMask = 0x08,           // Spectrum template trigger, available only in RTA mode. Not available in this stage.
        GNSS1PPS = 0x09				   // Use 1PPS provided by GNSS in the system to trigger.
    };

    public enum RTA_TriggerSource_TypeDef
    {
        FreeRun = 0x00,                // Free running
        External = 0x01,               // External trigger. Triggered by a physical signal connected to a trigger input port outside the device.
        Bus = 0x02,                    // The bus is triggered. Triggered by a function (instruction).
        Level = 0x03,                  // Level trigger. The device detects the input signal according to the set level threshold, and triggers automatically when the input exceeds the threshold.
        Timer = 0x04,                  // The timer is triggered. Use the device internal timer to automatically trigger the set time period.
        TxSweep = 0x05,                // Output trigger of signal generator scan; When this trigger source is selected, the acquisition process will be triggered by the output trigger signal scanned by the signal source.
        MultiDevSyncByExt = 0x06,      // When the external trigger signal arrives, multiple machines perform synchronous trigger behavior.
        MultiDevSyncByGNSS1PPS = 0x07, // When GNSS-1PPS arrives, multiple machines trigger synchronously.
        SpectrumMask = 0x08,           // Spectrum template trigger, available only in RTA mode. Not available in this stage.
        GNSS1PPS = 0x09				   // Use 1PPS provided by GNSS in the system to trigger.
    };

    /*The timer trigger is synchronized with the outer trigger edge*/
    public enum TriggerTimerSync_TypeDef
    {
        NoneSync = 0x00,                        // Timer trigger is not synchronized with the external trigger.
        SyncToExt_RisingEdge = 0x01,            // Timer trigger is synchronized with the external trigger rising edge. 
        SyncToExt_FallingEdge = 0x02,           // Timer trigger is synchronized with the external trigger falling edge.
        SyncToExt_SingleRisingEdge = 0x03,      // Timer trigger and external trigger rise edge single synchronization (need to call instruction function, single synchronization).
        SyncToExt_SingleFallingEdge = 0x04,     // Timer trigger and external trigger fall edge single synchronization (need to call instruction function, single synchronization).
        SyncToGNSS1PPS_RisingEdge = 0x05,       // Timer trigger synchronizes with the rising edge of GNSS-1PPS. 
        SyncToGNSS1PPS_FallingEdge = 0x06,      // Timer trigger synchronizes with the falling edge of GNSS-1PPS.
        SyncToGNSS1PPS_SingleRisingEdge = 0x07, // Timer triggers single synchronization with GNSS-1PPS rising edge (need to call instruction function, perform single synchronization).
        SyncToGNSS1PPS_SingleFallingEdge = 0x08 // Timer triggers single synchronization with GNSS-1PPS falling edge (need to call instruction function, perform single synchronization).
    };

    /*DC suppression(IQS\DET\RTA)*/
    public enum DCCancelerMode_TypeDef
    {
        DCCOff = 0x00,                // Disable the DC suppression function.
        DCCHighPassFilterMode = 0x01, // Open, high-pass filter mode. This mode has a good DC suppression effect, but will damage the signal component from DC to low frequency (about 100kHz).
        DCCManualOffsetMode = 0x02,   // Open, manual bias mode. In this mode, the bias value needs to be specified manually, and the suppression effect is weaker than that of the high-pass filter mode, but the signal component on the DC will not be damaged.
        DCCAutoOffsetMode = 0x03	  // Open, automatic bias mode.
    };

    /*Quadrature demodulation correction(IQS\DET\RTA)*/
    public enum QDCMode_TypeDef
    {
        QDCOff = 0x00,        // Close the QDC.
        QDCManualMode = 0x01, // Manually perform QDC based on the given parameters.
        QDCAutoMode = 0x02	  // Automatic QDC: The system automatically performs the calibration and uses the parameters obtained during each IQS_Configuration call.
    };

    /*Fixed frequency point mode (IQS\RTA\DET) trigger mode*/
    public enum TriggerMode_TypeDef
    {
        FixedPoints = 0x00, // Get the fixed point length data.
        Adaptive = 0x01		// Get data continuously.
    };

    /*Stream mode data format type(IQS/DSP)*/
    public enum DataFormat_TypeDef
    {
        Complex16bit = 0x00, // IQ, single channel data 16 bits.
        Complex32bit = 0x01, // IQ, single channel data 32 bits.
        Complex8bit = 0x02,  // IQ, single channel data 8 bits.
        Complexfloat = 0x06  // IQ, single channel data 32-bit float format tpye (IQS mode is not available, only by DDC function output data to write back the enumeration variable).
    };



    /*Source operating mode (TRx series only)(SIG)*/
    public enum SIG_OperationMode_TypeDef
    {
        SIG_PowerOff = 0x00,             // Close.
        SIG_Fixed = 0x01,                // Fixed frequency and power mode.
        SIG_FreqSweep_Single = 0x02,     // Single frequency scan (not available).
        SIG_FreqSweep_Continous = 0x03,  // Continuous frequency scan (not available).
        SIG_PowerSweep_Single = 0x04,    // Single power scan (not available).
        SIG_PowerSweep_Continous = 0x05, // Continuous power scan (not available).
        SIG_ListSweep_Single = 0x06,     // Single list scan (not available).
        SIG_ListSweep_Continous = 0x07,  // Continuous list scan (not available).
        SIG_ManualGainCtrl = 0x08		 // Manual gain control.
    };

    /*signal generator Scan Source (TRx Series only)(SIG)*/
    public enum SIG_SweepSource_TypeDef
    {
        SIG_SWEEPSOURCE_RF = 0x00,   // Radio frequency scan.
        SIG_SWEEPSOURCE_LF = 0x01,   // Low frequency scan.
        SIG_SWEEPSOURCE_LF_RF = 0x02 // Low frequency and RF co-scan.
    };

    /*Launch port Status in Launch Mode (TRx Series only)(SIG)*/
    public enum TxPort_TypeDef
    {
        RF_ExternalPort = 0x00, // External port.
        RF_InternalPort = 0x01, // Internal port.
        RF_ANT_Port = 0x02,     // ANT port (TRx series only).
        RF_TR_Port = 0x03,      // TR port (TRx series only).
        RF_SWR_Port = 0x04,     // SWR port (TRx series only).
        RF_INT_Port = 0x05		// INT port (TRx series only).
    };

    public enum RFPort_typedef
    {
        RF_ExternalPort = 0x00, // External port.
        RF_InternalPort = 0x01, // Internal port.
        RF_ANT_Port = 0x02,        // ANT port (TRx series only).
        RF_TR_Port = 0x03,        // TR port (TRx series only).
        RF_SWR_Port = 0x04,        // SWR port (TRx series only).
        RF_INT_Port = 0x05        // INT port (TRx series only).
    };

    /*Transmit baseband Transmission Reset State (TRx series only)(SIG)*/
    public enum TxTransferReset_TypeDef
    {
        Tx_TRNASFERRESET_ON = 0x00,  // The transmit transmission channel reset is enabled during configuration.
        Tx_TRNASFERRESET_OFF = 0x01 // The transmit transmission channel reset is disabled during configuration.
    };

    /*Transmit transmission link packet state*/
    public enum TxPackingCmd_TypeDef
    {
        TxPacking_Off = 0x00, // The transmit link packets are closed and valid data are transmitted directly
        TxPacking_On = 0x01   // The middle packet of the transmitter is opened and the packet header and packet tail are added for the valid data
    };

    /*The bit width of the transmit transmission link packet*/
    public enum TxPackingDataWidth_TypeDef
    {
        TxPackingDataWidth_32Bits = 0x01,  // When the packet is opened, the data bit width of the packet is 32bits, and the 32bits are aligned
        TxPackingDataWidth_64Bits = 0x02,  // When the packet is opened, the data bit width of the packet is 64bits, and the 64bits are aligned
        TxPackingDataWidth_128Bits = 0x04, // When the packet is opened, the data bit width of the packet is 128bits, and the 128bits are aligned
        TxPackingDataWidth_256Bits = 0x08  // When the packet is opened, the data bit width of the packet is 256bits, and the 256bits are aligned
    };

    /*Trigger input source in emission mode (TRx series only)(SIG)*/
    public enum TxTriggerInSource_TypeDef
    {
        Tx_TRIGGERIN_SOURCE_INTERNAL = 0x00, // Internal trigger.
        Tx_TRIGGERIN_SOURCE_EXTERNAL = 0x01, // External trigger.
        Tx_TRIGGERIN_SOURCE_TIMER = 0x02,    // The timer trigger.
        Tx_TRIGGERIN_SOURCE_RF = 0x03,       // RF board trigger.
        Tx_TRIGGERIN_SOURCE_BUS = 0x04       // The bus trigger.
    };

    /*Trigger input Mode in Emission Mode (TRx series only)(SIG)*/
    public enum TxTriggerInMode_TypeDef
    {
        Tx_TRIGGER_MODE_FREERUN = 0x00,     // Free running.
        Tx_TRIGGER_MODE_SINGLEPOINT = 0x01, // Single point trigger (trigger a single frequency or power configuration).
        Tx_TRIGGER_MODE_SINGLESWEEP = 0x02, // Single scan trigger (trigger one cycle scan at a time).
        Tx_TRIGGER_MODE_CONTINOUS = 0x03    // Continuous scan trigger (trigger a continuous work).
    };

    /*Trigger Output Mode in Emission Mode (TRx Series only)(SIG)*/
    public enum TxTriggerOutMode_TypeDef
    {
        Tx_TRIGGER_OUTMODE_NONE = 0x00,      // No output trigger.
        Tx_TRIGGER_OUTMODE_PERPOINT = 0x01,  // Single point output (single trigger output for single frequency hopping and power hopping).
        Tx_TRIGGER_OUTMODE_PERSWEEP = 0x02,  // Output a single scan (output a trigger after the completion of a single scan cycle).
        Tx_TRIGGER_OUTMODE_PERPROFILE = 0x03 // Single Profile output (output a trigger after a single Profile scan is completed).
    };

    /*Analog IQ Input Source in Emission Mode (TRx Series only)(SIG)*/
    public enum TxAnalogIQSource_TypeDef
    {
        Tx_ANALOGIQSOURCE_INTERNAL = 0x00, // Internal simulated IQ input.
        Tx_ANALOGIQSOURCE_EXTERNAL = 0x01  // External simulated IQ input.
    };

    /*Digital IQ input source in Emission Mode (TRx Series only)(SIG)*/
    public enum TxDigitalIQSource_TypeDef
    {
        Tx_DIGITALIQSOURCE_INTERNAL = 0x00,  // Internal digital IQ input.
        Tx_DIGITALIQSOURCE_EXTERNAL = 0x01,  // External digital IQ input.
        Tx_DIGITALIQSOURCE_SIMULATION = 0x02  //Use internal sinusoidal simulation signals.
    };

    /*Trigger input source in Send/Receive mode (TRx Series only)(VNA)*/
    public enum TRxTriggerInSource_typedef
    {
        TRx_TRIGGERIN_SOURCE_INTERNAL = 0x00, // Internal trigger.
        TRx_TRIGGERIN_SOURCE_EXTERNAL = 0x01, // External trigger.
        TRx_TRIGGERIN_SOURCE_TIMER = 0x02,    // The timer trigger.
        TRx_TRIGGERIN_SOURCE_RF = 0x03,       // RF board trigger.
        TRx_TRIGGERIN_SOURCE_BUS = 0x04       // Bus trigger.
    };

    /*Trigger input source in Send/Receive mode (TRx Series only)(VNA)*/
    public enum TRxTriggerInMode_typedef
    {
        TRx_TRIGGER_MODE_FREERUN = 0x00,     // Free running.
        TRx_TRIGGER_MODE_SINGLEPOINT = 0x01, // Single point trigger (trigger a single frequency or power configuration).
        TRx_TRIGGER_MODE_SINGLESWEEP = 0x02, //  Single scan trigger (trigger one cycle scan at a time).
        TRx_TRIGGER_MODE_CONTINOUS = 0x03    // Continuous scan trigger (trigger a continuous work).
    };

    /*Trigger input mode in Send/Receive Mode (Only for TRx Series)(VNA)*/
    public enum TRxTriggerOutMode_typedef
    {
        TRx_TRIGGER_OUTMODE_NONE = 0x00,      // No trigger output
        TRx_TRIGGER_OUTMODE_PERPOINT = 0x01,  // Single point trigger (trigger a single frequency or power configuration).
        TRx_TRIGGER_OUTMODE_PERSWEEP = 0x02,  // Single scan trigger (trigger one cycle scan at a time).
        TRx_TRIGGER_OUTMODE_PERPROFILE = 0x03 // Single Profile output (a trigger is generated after each Profile scan is completed)
    };

    /*Status of the sending and receiving ports in the sending and receiving mode: The former is sending and the latter is receiving (Only applicable to TRx series)(VNA)*/
    public enum TRxPort_typedef
    {
        ANT_TR = 0x00,   // The transmit is ANT and the receive is TR.  	
        TR_ANT = 0x01,   // The transmit is TR and the receive is ANT.
        SWR_SWR = 0x02,  // The transmit is SWR and the receive is SWR.
        INT_INT = 0x03,  // The transmit is INT and the receive is INT.(internal to internal))
        ANT_ANT = 0x04,  // The transmit is ANT and the receive is ANT.
        SWR_ANT = 0x05,  // The transmit is SWR and the receive is ANT.
        SWR_TR = 0x06,   // The transmit is SWR and the receive is TR.
        TR_SWR = 0x07,   // The transmit is TR and the receive is SWR.
        SWR = 0x08,      // Used specifically to refer to standing wave (S11) test.
        TRANSMIT = 0x09  // Dedicated to transport (S21) test, for TRX60 =SWR->ANT; For SAM-60 = Tx->Rx.
    };

    /*Receive reference Level Mode in Send/Receive Mode (TRx Series only)(VNA)*/
    public enum RxLevelMode_typedef
    {
        RxRefLevel_Fixed = 0x00, // Set the reference level to CurrentLevel_dBm.
        RxRefLevel_Sync = 0x01  // Changes synchronously with the transmit power, and the offset is CurrentLevel_dBm.
    };

    /*Vector Network Analyzer working mode (only for TRx series)(VNA)*/
    public enum VNA_OperationMode_TypeDef
    {
        Sweep_S11 = 0x00,   // S11 mode to measure reflection characteristics.
        Sweep_S21 = 0x01,   // S21 mode to measure transmission characteristics.
        Sweep_Power = 0x02, // Power scanning mode, measuring power compression point, etc.
        Fixed = 0x03        // Fixed frequency and power mode.
    };

    /*Vector Network Analyzer Calibration Mode (TRx Series only)(VNA)*/
    public enum VNA_Calibration_TypeDef
    {
        Calibration_Open = 0x00,     // In S11 mode, open circuit calibration.	
        Calibration_Short = 0x01,    // In S11 mode, short circuit calibration.		
        Calibration_Match = 0x02,    // In S11 mode, match calibration.	
        Calibration_Through = 0x03,  // S21 mode, through calibration.
        Calibration_Isolation = 0x04 // Isolated calibration in S21 mode.
    };

    /*Signal type, used to generate the signal to be modulated (Analog) for analog modulation*/
    public enum AnalogSignalType_TypeDef
    {
        SIGNAL_SINE = 0x00,     // Sinusoidal signal.
        SIGNAL_COSINE = 0x01,   // Cosine signal.
        SIGNAL_LINEAR = 0x02,   // linear (sawtooth).
        SIGNAL_TRIANGLE = 0x03, // Triangle.
        SIGNAL_CW = 0x04,       // DC (output continuous wave after modulation).
        SIGNAL_COMPLEX = 0x05   // complex signal
    };

    /*AM modulation type (Analog)*/
    public enum AMModType_TypeDef
    {
        AM_MOD_DSB = 0x00, // double sideband modulation.
        AM_MOD_LSB = 0x01, // lower band modulation.
        AM_MOD_USB = 0x02, // upper band modulation.
        AM_MOD_CSB = 0x03, // Residual sideband modulation.
        AM_MOD_CW = 0x04   // Generate continuous waves.
    };

    /*AM modulated carrier suppression (Analog)*/
    public enum AM_CarrierSuppression_Typedef
    {
        AM_CARRIERSUPPRESSION_ON = 0x00, // Carrier suppression is on.
        AM_CARRIERSUPPRESSION_OFF = 0x01 // Carrier suppression is off.
    };

    /*Frequency detection mode (DSP)*/
    public enum TraceFormat_TypeDef
    {
        TraceFormat_Standard = 0x00, // equal frequency interval.
        TraceFormat_PrecisFrq = 0x01 // precise frequency.
    };

    /*Fan mode(DEV)*/
    public enum FanState_TypeDef
    {
        FanState_On = 0x00,     // forced on.
        FanState_Off = 0x01,    // forced off.
        FanState_Auto = 0x02	// automatically.
    };

    /*GNSS Antenna status*/
    public enum GNSSAntennaState_TypeDef
    {
        GNSS_AntennaExternal = 0x00,  // external antenna
        GNSS_AntennaInternal = 0x01 // inter antenna
    };

    /*DOCXO Antenna static_assert*/
    public enum DOCXOWorkMode_TypeDef
    {
        DOCXO_LockMode = 0x00,  // disciplining mode
        DOCXO_HoldMode = 0x01 	// tracking mode
    };

    //GNSS signal-to-noise ratio (SNR)
    [StructLayout(LayoutKind.Sequential)]
    public struct GNSS_SNR_TypeDef
    {
        public uint8_t Max_SatxC_No;   //maximum signal-to-noise ratio (SNR)
        public uint8_t Min_SatxC_No;   //minimum signal-to-noise ratio (SNR)
        public uint8_t Avg_SatxC_No;	//average signal-to-noise ratio (SNR)
    };

    [StructLayout(LayoutKind.Sequential)]
    public struct GNSS_SatDate_TypeDef
    {
        public uint8_t SatsNum_All;    //current range visual satellite
        public uint8_t SatsNum_Use;    //number of satellites used for positioning
        public GNSS_SNR_TypeDef GNSS_SNR_UsePos;       //satellite signal-to-noise ratio information for positioning
        public GNSS_SNR_TypeDef GNSS_SNR_NotUsePos;	//satellite signal-to-noise ratio information in view, but not used for positioning
    };


    /*GNSS obtain device information*/
    [StructLayout(LayoutKind.Sequential)]
    public struct GNSSInfo_TypeDef
    {

        public float latitude;        //longitude
        public float longitude;         //lattitude
        public int16_t altitude;           //altitude
        public uint8_t SatsNum;             //number of satellites currently in use

        public uint8_t GNSS_LockState;     //GPS lock state 
        public uint8_t DOCXO_LockState;    //GDOCXO lock state
        public DOCXOWorkMode_TypeDef DOCXO_WorkMode;     //DOCXO work state
        public GNSSAntennaState_TypeDef GNSSAntennaState; //Antenna sta

        public int16_t hour;
        public int16_t minute;
        public int16_t second;
        public int16_t Year;
        public int16_t month;
        public int16_t day;
    };

    // GNSS Peripheral Type
    public enum GNSSPeriphType_TypeDef
    {
        GNSS_None = 0,        // No Peripherals
        GNSS_For_EIO = 1,     // EIO
        GNSS_For_NX = 2,      // NX
        GNSS_For_PX = 3,      // PX
        GNSS_For_PXZ = 4      // PXZ
    }

    // GNSS Type
    public enum GNSSType_TypeDef
    {
        None_GPS = 0, // no GPS receiver
        GNSS_GPS = 1, // Standard GPS
        GNSS_GPS_Pro = 2 // High Performance GPS
    }

    // OCXO Type
    public enum OCXOType_TypeDef
    {
        None_OCXO = 0, // None OCXO
        GNSS_OCXO = 1, // normal OCXO on GNSS
        GNSS_DOCXO = 2 // tameable OCXO on GNSS
    }

    // Hardware State Structure
    [StructLayout(LayoutKind.Sequential)]
    public struct HardWareState_TypeDef
    {
        public GNSSPeriphType_TypeDef GNSSPeriphType; // GNSS peripheral type
        public GNSSType_TypeDef GNSSType; // GNSS receiver type
        public OCXOType_TypeDef OCXOType; // OCXO type on GNSS

        public uint8_t InternalOCXO; // Whether or not the device's internal reference clock is a thermostat crystal
        public uint8_t SignalSourceEn; // Whether the device supports signal source functionality
        public uint8_t ADC_VariableRateEn; // Whether the device supports ADC variable sample rate
        public uint8_t IM3_filter; // Supplemental IF filter (IM3 enhancement)
    }

    /*LO optimization(all)*/
    public enum LOOptimization_TypeDef
    {
        LOOpt_Auto = 0x00,      // LO optimization,auto
        LOOpt_Speed = 0x01,     // LO optimization,high sweep speed
        LOOpt_Spur = 0x02,      // LO optimization,low spurious
        LOOpt_PhaseNoise = 0x03	// LO optimization,low phase noise
    };



    /*---------------------------------------------------
    Device structure
    -----------------------------------------------------*/

    /*Startup configuration (Configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct BootProfile_TypeDef
    {
        public PhysicalInterface_TypeDef PhysicalInterface; // Select the physical interface.
        public DevicePowerSupply_TypeDef DevicePowerSupply; // Select the power supply mode.
        public IPVersion_TypeDef ETH_IPVersion;             // ETH Protocol version.
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 16)]
        public uint8_t[] ETH_IPAddress; // IP address of the ETH interface.
        public uint16_t ETH_RemotePort;                     // Listening port of the ETH interface.
        public int32_t ETH_ErrorCode;                         // Return code of the ETH interface.
        public int32_t ETH_ReadTimeOut;                     // Read timeout of the ETH interface(ms).
    };

    /*Device information (returned)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct DeviceInfo_TypeDef
    {
        public uint64_t DeviceUID;          // Device serial number.
        public uint16_t Model;              // Device type.
        public uint16_t HardwareVersion; // Hardware version.
        public uint32_t MFWVersion;      // MCU firmware version.
        public uint32_t FFWVersion;      // FPGA firmware version.
    };

    /*NX Device information (returned)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct NetworkDeviceInfo_TypeDef
    {
        public uint64_t DeviceUID;       // Device serial number.
        public uint16_t Model;           // Device type.
        public uint16_t HardwareVersion; // Hardware version.
        public uint32_t MFWVersion;      // MCU firmware version.
        public uint32_t FFWVersion;      // FPGA firmware version.
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
        public uint8_t[] IPAddress;     // IP Address.
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
        public uint8_t[] SubnetMask;    // Subnet Mask.
    };

    /*Startup information (return)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct BootInfo_TypeDef
    {
        public DeviceInfo_TypeDef DeviceInfo; // Device information.

        public uint32_t BusSpeed;               // Bus speed information.
        public uint32_t BusVersion;           // Bus firmware version.
        public uint32_t APIVersion;           // API version

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 7)]
        public int[] ErrorCodes; //  // List of error codes during startup.
        public int Errors;                       // Total number of errors during startup.
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 7)]
        public int[] WarningCodes; // List of warning codes during startup.
        public int Warnings;                   // Total number of warnings during startup.
    };

    /*Device status (advanced variable, not recommended)(Return)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct DeviceState_TypeDef
    {
        public int16_t Temperature;      // Equipment Temperature, Celsius = 0.01 * Temperature.
        public uint16_t RFState;         // Radio status.
        public uint16_t BBState;        // Baseband status.

        public double AbsoluteTimeStamp; // The absolute timestamp of the current packet.
        public float Latitude;           // Latitude coordinates corresponding to the current packet. North latitude is positive and south latitude is negative, so as to distinguish north and south latitudes.
        public float Longitude;          // The longitude coordinate corresponding to the current packet is positive in east longitude and negative in west longitude, so as to distinguish east longitude from west longitude.

        public uint16_t GainPattern;      // Gain control word used by the frequency point of the current packet.
        public int64_t RFCFreq;          // RF center frequency used by the frequency point of the current packet.

        public uint32_t ConvertPattern;  // Frequency conversion mode used by the frequency point of the current packet.
        public uint32_t NCOFTW;          // NCO frequency word used by the current packet frequency point.

        public uint32_t SampleRate;      // Equivalent sampling rate used by the current packet frequency point, equivalent sampling rate = ADC sampling rate/extraction multiple.
        public uint16_t CPU_BCFlag;      // CPU-FFT Specifies the BC flag bit required for the frame.
        public uint16_t IFOverflow;      // If the equipment is overloaded, consider and BBState or RFState.
        public uint16_t DecimateFactor;  // The extraction multiple used by the current packet frequency point.
        public uint16_t OptionState;     // Optional status.

        //int16_t        LicenseCode;   //  License code.
    };

    /* Supplementary information for measurement data */
    [StructLayout(LayoutKind.Sequential)]
    public struct MeasAuxInfo_TypeDef
    {
        public uint32_t MaxIndex;            // Indicates the index of the maximum power value in the current packet.
        public float MaxPower_dBm;        // The maximum power in the current packet.

        public int16_t Temperature;        // Equipment Temperature, Celsius = 0.01 * Temperature.
        public uint16_t RFState;            // Radio status.
        public uint16_t BBState;            // Baseband status.
        public uint16_t GainPattern;        // Gain control word used by the frequency point of the current packet.

        public uint32_t ConvertPattern;    // Frequency conversion mode used by the frequency point of the current packet.

        public double SysTimeStamp;        // System time stamp corresponding to the current packet, in s.

        public double AbsoluteTimeStamp;    // Absolute timestamp of the current packet.
        public float Latitude;            // Latitude coordinates corresponding to the current packet. North latitude is positive and south latitude is negative, so as to distinguish north and south latitudes.
        public float Longitude;            // The longitude coordinate corresponding to the current packet is positive in east longitude and negative in west longitude, so as to distinguish east longitude from west longitude.

    };

    /*---------------------------------------------------
    Standard spectrum mode(SWP)
    -----------------------------------------------------*/
    /*SWP configuration structure (basic configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct SWP_EZProfile_TypeDef
    {
        public double StartFreq_Hz;                                 // start frequency
        public double StopFreq_Hz;                                    // Stop frequency
        public double CenterFreq_Hz;                              // center frequency
        public double Span_Hz;                                    // span
        public double RefLevel_dBm;                               // R.L.
        public double RBW_Hz;                                        // RBW
        public double VBW_Hz;                                       // VBW
        public double SweepTime;                                   // When the sweep time mode is Manual, the parameter is absolute time. When specified as *N, this parameter is the scan time multiplier

        public SWP_FreqAssignment_TypeDef FreqAssignment;         // Select StartStop or CenterSpan to set the frequency.

        public Window_TypeDef Window;                                // Window type used for FFT analysis.

        public RBWMode_TypeDef RBWMode;                             // RBW update mode. Input manually, set automatically according to Span.

        public VBWMode_TypeDef VBWMode;                           // VBW update mode. Manual input, VBW = RBW, VBW = 0.1*RBW, VBW = 0.01*RBW.

        public SweepTimeMode_TypeDef SweepTimeMode;               // sweep time mode

        public Detector_TypeDef Detector;                          // Detector

        public TraceDetectMode_TypeDef TraceDetectMode;           // trace detection mode(frequency axis)
        public TraceDetector_TypeDef TraceDetector;                // trace detector

        public uint32_t TracePoints;                                // trace points

        public RxPort_TypeDef RxPort;                                 // RF in port

        public SpurRejection_TypeDef SpurRejection;               // spurious rejection

        public ReferenceClockSource_TypeDef ReferenceClockSource;  // reference clock source
        public double ReferenceClockFrequency;                    // reference clock frequency,Hz

        public SWP_TriggerSource_TypeDef TriggerSource;            // input trigger source.
        public TriggerEdge_TypeDef TriggerEdge;                      // input trigger edge.

        public PreamplifierState_TypeDef Preamplifier;               // Preamplifier action Settings

        public SWP_TraceType_TypeDef TraceType;                    // output trace type

    };

    /*Standard spectrum configuration structure (configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct SWP_Profile_TypeDef
    {
        public double StartFreq_Hz;                                      // Start frequency.
        public double StopFreq_Hz;                                          // Stop frequency
        public double CenterFreq_Hz;                                  // Center frequency
        public double Span_Hz;                                          // Span
        public double RefLevel_dBm;                                    // Reference level
        public double RBW_Hz;                                              // RBW
        public double VBW_Hz;                                              // VBW
        public double SweepTime;                                         // When the sweep time mode is Manual, the parameter is absolute time. When specified as *N, this parameter is the scan time multiplier
        public double TraceBinSize_Hz;                                   // The frequency interval between adjacent frequency points of the trace.

        public SWP_FreqAssignment_TypeDef FreqAssignment;              // Select StartStop or CenterSpan to set the frequency.

        public Window_TypeDef Window;                                       //  Window type used for FFT analysis.

        public RBWMode_TypeDef RBWMode;                                   // RBW update mode. Input manually, set automatically according to Span.

        public VBWMode_TypeDef VBWMode;                                   // VBW update mode. Manual input, VBW = RBW, VBW = 0.1*RBW, VBW = 0.01*RBW.

        public SweepTimeMode_TypeDef SweepTimeMode;                     // sweep time mode.

        public Detector_TypeDef Detector;                                  // detector

        public TraceFormat_TypeDef TraceFormat;                          // Trace format
        public TraceDetectMode_TypeDef TraceDetectMode;                 // trace detection mode (frequency axis)
        public TraceDetector_TypeDef TraceDetector;                       // trace detector

        public uint32_t TracePoints;                                        // Trace point.
        public TracePointsStrategy_TypeDef TracePointsStrategy;          // Trace point mapping strategy.
        public TraceAlign_TypeDef TraceAlign;                          // Trace alignment.

        public FFTExecutionStrategy_TypeDef FFTExecutionStrategy;          // FFT execution strategy.

        public RxPort_TypeDef RxPort;                                      	 // RF input port.

        public SpurRejection_TypeDef SpurRejection;                       // Spurious Rejection

        public ReferenceClockSource_TypeDef ReferenceClockSource;          // Reference clock source.
        public double ReferenceClockFrequency;                           // Reference clock frequency, Hz.
        public uint8_t EnableReferenceClockOut;                         // enable reference clock output.

        public SystemClockSource_TypeDef SystemClockSource;               // The default system clock source is the internal system clock.Use it under the guidance of the vendor.
        public double ExternalSystemClockFrequency;                         	 // External system clock frequency (Hz).

        public SWP_TriggerSource_TypeDef TriggerSource;                 // Input trigger source.
        public TriggerEdge_TypeDef TriggerEdge;                           // Input trigger edge.
        public TriggerOutMode_TypeDef TriggerOutMode;                     // Trigger output mode.
        public TriggerOutPulsePolarity_TypeDef TriggerOutPulsePolarity; // Trigger output pulse polarity.

        public uint32_t PowerBalance;                                   // Balance between power consumption and scanning speed.
        public GainStrategy_TypeDef GainStrategy;                          // Gain strategy.
        public PreamplifierState_TypeDef Preamplifier;                      // Preamplifier action set.

        public uint8_t AnalogIFBWGrade;                                  //	Trigger output mode.
        public uint8_t IFGainGrade;                                        //	Trigger output pulse polarity.

        public uint8_t EnableDebugMode;                                 //	Debug mode. Advanced applications are not recommended. The default value is 0.
        public uint8_t CalibrationSettings;                               // Calibration selection. Advanced applications are not recommended. The default value is 0.

        public int8_t Atten;                                             // Attenuation dB, set the frequency analyzer channel attenuation, default -1 (automatic).

        public SWP_TraceType_TypeDef TraceType;                      	 // Output trace type.

        public LOOptimization_TypeDef LOOptimization;                      // LO optimization

    };

    /*Trace information structure of SWP mode (return)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct SWP_TraceInfo_TypeDef
    {
        public int FullsweepTracePoints;         // The points of the complete trace.
        public int PartialsweepTracePoints;     // Trace points of each frequency point, that is, the points of GetPart each time.
        public int TotalHops;                     // The number of frequency points of complete traces, that is, the number of times a complete trace needs GetPart
        public uint32_t UserStartIndex;         // Array index corresponding to the user-specified StartFreq_Hz in the trace array, that is, when HopIndex = 0, Freq[UserStartIndex] is the closest frequency point to SWPProfile.StartFreq_Hz.
        public uint32_t UserStopIndex;            // Array index corresponding to the user-specified StopFreq_Hz in the trace array, that is, when HopIndex = TotalHops-1, Freq[UserStopIndex] is the frequency point closest to SWPProfile.StopFreq_Hz.

        public double TraceBinBW_Hz;             // The frequency interval between two points of the trace.
        public double StartFreq_Hz;             // The frequency of the first frequency point of the trace.
        public double AnalysisBW_Hz;              // Analysis bandwidth corresponding to each frequency point.

        public int TraceDetectRatio;             // Detection ratio of video detection.
        public int DecimateFactor;                  // Multiple of time domain data extraction.
        public float FrameTimeMultiple;         // Frame analysis time multiple: The analysis time of the device at a single frequency = default analysis time (set by the system) * frame time multiple. Increasing the frame time multiple will increase the device's minimum scan time, but is not strictly linear.
        public double FrameTime;                   // Frame sweep time: duration (in seconds) of the signal used for single frame FFT analysis.
        public double EstimateMinSweepTime;      // The minimum scanning time that can be set under the current configuration (unit: second, the result is mainly affected by Span, RBW, VBW, frame scanning time and other factors). 
        public DataFormat_TypeDef DataFormat;   // Time domain data format.
        public uint64_t SamplePoints;            // Time domain data sampling length.
        public uint32_t GainParameter;             // Gain related parameters, including Space(31 to 24Bit), PreAmplifierState(23 to 16Bit), StartRFBand(15 to 8Bit), StopRFBand(7 to 0Bit).
        public DSPPlatform_Typedef DSPPlatform; // DSP calculating platform under current configuration
    };

    /*Debug information structure for scan mode (Return)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct SWP_DebugInfo_TypeDef
    {
        public double RFCFreq;                	//
        public double RFLOFreq;                	//
        public double IFLOFreq;                	//
        public double IF1STFreq;                //
        public double IF2NDFreq;                //
        public double NCOFreq;                	//

        public int HopIndex;                	//

        public uint8_t RFBand;                  //
        public uint8_t HighSideRFLO;            //
        public uint8_t HighSideIFLO;            //
        public uint8_t IQInvert;                //

        public uint8_t RFGainSpace;            	//
        public uint8_t RFGainGrade;            	//
        public uint8_t AnalogIFBWGrade;        	//
        public uint8_t IFGainGrade;            	//

        public uint16_t RFState;                //
        public uint16_t BBState;                //

        public uint32_t LowCTIndex;            	//
        public uint32_t HighCTIndex;            //
        public uint32_t RTIndex;                //

        public int16_t CalibratedTemperature;  	//
        public int16_t LowCharactTemperature;  	//
        public int16_t HighCharactTemperature; 	//
        public int16_t ReferenceTemperature;    //

        public float RFACalVal;                	//
    };

    /*---------------------------------------------------
   IQS mode structure(IQStream)
    -----------------------------------------------------*/
    /*IQS configuration structure (basic configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct IQS_EZProfile_TypeDef
    {
        public double CenterFreq_Hz;                              	 // Center frequency.
        public double RefLevel_dBm;                            	 // R.L.
        public uint32_t DecimateFactor;                        	 // Decimate Factor of time domain.

        public RxPort_TypeDef RxPort;                        	 // RF input port.

        public uint32_t BusTimeout_ms;                              // Transmission timeout

        public IQS_TriggerSource_TypeDef TriggerSource;         // Input trigger source.
        public TriggerEdge_TypeDef TriggerEdge;                   // Input trigger edge.

        public TriggerMode_TypeDef TriggerMode;                   // Input trigger mode.	
        public uint64_t TriggerLength;                               // Enter the number of sampling points after triggering. take effect only in FixedPoints mode.

        public double TriggerLevel_dBm;                            // Level trigger threshold.
        public double TriggerTimer_Period;                           // timing trigger,trigger period in s.		

        public DataFormat_TypeDef DataFormat;                      // data format.

        public PreamplifierState_TypeDef Preamplifier;               // Preamplifier action.

        public uint8_t AnalogIFBWGrade;                          //	IF gain grade.

        public ReferenceClockSource_TypeDef ReferenceClockSource; //	Reference clock source.
        public double ReferenceClockFrequency;                       //	Reference clock frequency.

        public double NativeIQSampleRate_SPS;                       // for specific equipment. Native IQ sampling rate. For devices with variable sampling rate, the sampling rate can be adjusted by this parameter; device with fixed sample rate are configured to default fixed value.

        public uint8_t EnableIFAGC;                               // for specific equipment. IF AGC control, 0: AGC off, using MGC mode; 1: The AGC is enabled.

    };

    /*Configuration structure of IQS (Configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct IQS_Profile_TypeDef
    {
        public double CenterFreq_Hz;                                     // Center frequency.
        public double RefLevel_dBm;                                       // Reference level.
        public uint32_t DecimateFactor;                                // Decimate Factor of time domain.

        public RxPort_TypeDef RxPort;                                      // RF input port.

        public uint32_t BusTimeout_ms;                                     // Transmission timeout

        public IQS_TriggerSource_TypeDef TriggerSource;                  // Input trigger source.
        public TriggerEdge_TypeDef TriggerEdge;                         // Input trigger edge.
        public TriggerMode_TypeDef TriggerMode;                         // Input trigger mode.	
        public uint64_t TriggerLength;                                     // Enter the number of sampling points after triggering. This takes effect only in FixedPoints mode.

        public TriggerOutMode_TypeDef TriggerOutMode;                     // Trigger output mode.
        public TriggerOutPulsePolarity_TypeDef TriggerOutPulsePolarity; // Trigger output pulse polarity.

        public double TriggerLevel_dBm;                                // Level trigger threshold.
        public double TriggerLevel_SafeTime;                             // Level trigger anti-shaking safety time, in seconds.
        public double TriggerDelay;                                     // Trigger delay,in seconds
        public double PreTriggerTime;                                     // Pre-trigger time,in seconds.

        public TriggerTimerSync_TypeDef TriggerTimerSync;                  // Synchronization options for timed and out-triggered edges. The trigger mode is effective when the trigger is timed.
        public double TriggerTimer_Period;                                 // The trigger period of the timed trigger, in s. The trigger mode is effective when the trigger is timed.

        public uint8_t EnableReTrigger;                                // Enable the device to respond multiple times after capturing a trigger. This function is available only in FixedPoint mode.
        public double ReTrigger_Period;                                 // Time interval for multiple responses of a trigger device. It is also used as the trigger period in the Timer trigger mode (unit: s).
        public uint16_t ReTrigger_Count;                                   // After a trigger, several responses are required in addition to the triggered response.

        public DataFormat_TypeDef DataFormat;                              // Data format.

        public GainStrategy_TypeDef GainStrategy;                          // Gain strategy.
        public PreamplifierState_TypeDef Preamplifier;                     // Preamplifier action.

        public uint8_t AnalogIFBWGrade;                                  //	Intermediate frequency bandwidth grade.
        public uint8_t IFGainGrade;                                      //	intermediate frequency gain grade.

        public uint8_t EnableDebugMode;                                 //	Debug mode. Advanced applications are not recommended. The default value is 0.

        public ReferenceClockSource_TypeDef ReferenceClockSource;         //	Reference clock source.
        public double ReferenceClockFrequency;                             // Reference clock frequency.
        public uint8_t EnableReferenceClockOut;                         // enable reference clock output.

        public SystemClockSource_TypeDef SystemClockSource;                // System clock source.
        public double ExternalSystemClockFrequency;                        // External system clock frequency (Hz).

        public double NativeIQSampleRate_SPS;                              // Suitable for specific equipment. Native IQ sampling rate. For devices with variable sampling rate, the sampling rate can be adjusted by adjusting this parameter; Nonvariable sampling rate device configurations are always configured to the system default fixed value.

        public uint8_t EnableIFAGC;                                  // Suitable for specific equipment. Medium frequency AGC control, 0: AGC off, using MGC mode; 1: The AGC is enabled.

        public int8_t Atten;                                          // attenuation.

        public DCCancelerMode_TypeDef DCCancelerMode;                     // Suitable for specific equipment. Dc suppression. 0: disables the DCC. 1: Open, high-pass filter mode (better suppression effect, but will damage the signal in the range of DC to 100 KHZ); 2: Open, manual bias mode (need manual calibration, but not low frequency damage signal).

        public QDCMode_TypeDef QDCMode;                                 // Suitable for specific equipment. IQ amplitude and phase corrector. QDCOff: disables the QDC function. QDCManualMode: Enable and use manual mode; QDCAutoMode: Enables and uses the automatic QDC mode.

        public float QDCIGain;                                             // Suitable for specific equipment. Normalized linear gain I, 1.0 indicates no gain, set range 0.8 to 1.2.
        public float QDCQGain;                                              // Suitable for specific equipment. Normalized linear gain Q, 1.0 indicates no gain, set range 0.8~1.2.
        public float QDCPhaseComp;                                        // Suitable for specific equipment. Normalized phase compensation coefficient, set range -0.2~+0.2.

        public int8_t DCCIOffset;                                         // Suitable for specific equipment. I channel DC offset, LSB.
        public int8_t DCCQOffset;                                         // Suitable for specific equipment. Q channel DC offset, LSB.

        public LOOptimization_TypeDef LOOptimization;                    // LO optimization

    };

    /* Flow information structure returned after IQS configuration (returned) */
    [StructLayout(LayoutKind.Sequential)]
    public struct IQS_StreamInfo_TypeDef
    {
        public double Bandwidth;        // The current configuration corresponds to the receiver's physical channel or digital signal processing bandwidth.
        public double IQSampleRate;     // Single-channel sampling rate of IQ corresponding to the current configuration (unit: Sample/second).
        public uint64_t PacketCount;     // The total number of data packets corresponding to the current configuration takes effect only in FixedPoints mode.

        public uint64_t StreamSamples;    // In Fixedpoints mode, it represents the total number of sampling points corresponding to the current configuration. In Adaptive mode, the value has no physical significance and is 0.
        public uint64_t StreamDataSize; // In Fixedpoints mode, it indicates the total number of bytes of samples corresponding to the current configuration. In Adaptive mode, the value has no physical significance and is 0.

        public uint32_t PacketSamples;  // Sampling points in packets obtained by each IQS_GetIQStream invocation Sample points contained in each packet.
        public uint32_t PacketDataSize;// The number of valid data bytes to be obtained per call to IQS_GetIQStream.
        public uint32_t GainParameter;     // Gain dependent parameters, including Space(31~24Bit),PreAmplifierState(23~16Bit),StartRFBand(15~8Bit)、StopRFBand(7~0Bit).
    };

    /* Trigger information structure contained in IQS packet, trigger information return structure of DET and RTA is the same (return) */
    [StructLayout(LayoutKind.Sequential)]
    public struct IQS_TriggerInfo_TypeDef
    {
        public uint64_t SysTimerCountOfFirstDataPoint;      // The system timestamp corresponding to the first data point of the current packet.
        public uint16_t InPacketTriggeredDataSize;          // The number of bytes of valid trigger data in the current packet.
        public uint16_t InPacketTriggerEdges;             // The number of trigger edges contained in the current package.
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 25)]
        public uint32_t[] StartDataIndexOfTriggerEdges; // The data location in the current package corresponding to the trigger edge.
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 25)]
        public uint64_t[] SysTimerCountOfEdges; // The system timestamp of the trigger edge in the current package.
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 25)]
        public int8_t[] EdgeType;  // The polarity of each trigger edge in the current packet.
    };

    [StructLayout(LayoutKind.Sequential)]
    public struct DET_TriggerInfo_TypeDef
    {
        public uint64_t SysTimerCountOfFirstDataPoint;      // The system timestamp corresponding to the first data point of the current packet.
        public uint16_t InPacketTriggeredDataSize;          // The number of bytes of valid trigger data in the current packet.
        public uint16_t InPacketTriggerEdges;             // The number of trigger edges contained in the current package.
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 25)]
        public uint32_t[] StartDataIndexOfTriggerEdges; // The data location in the current package corresponding to the trigger edge.
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 25)]
        public uint64_t[] SysTimerCountOfEdges; // The system timestamp of the trigger edge in the current package.
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 25)]
        public int8_t[] EdgeType;  // The polarity of each trigger edge in the current packet.
    };

    [StructLayout(LayoutKind.Sequential)]
    public struct RTA_TriggerInfo_TypeDef
    {
        public uint64_t SysTimerCountOfFirstDataPoint;      // The system timestamp corresponding to the first data point of the current packet.
        public uint16_t InPacketTriggeredDataSize;          // The number of bytes of valid trigger data in the current packet.
        public uint16_t InPacketTriggerEdges;             // The number of trigger edges contained in the current package.
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 25)]
        public uint32_t[] StartDataIndexOfTriggerEdges; // The data location in the current package corresponding to the trigger edge.
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 25)]
        public uint64_t[] SysTimerCountOfEdges; // The system timestamp of the trigger edge in the current package.
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 25)]
        public int8_t[] EdgeType;  // The polarity of each trigger edge in the current packet.
    };

    /* IQS Top-level structures containing configuration, return information, and staging data (configuration, return, staging) */
    [StructLayout(LayoutKind.Sequential)]
    public struct IQStream_TypeDef
    {
        public IntPtr AlternIQStream;                      // Interleaved distribution of IQ time domain data, single path may be in int-8, int-16 and int-32 format.

        public float IQS_ScaleToV;                          // Coefficient from type int to absolute value of voltage (V).

        public float MaxPower_dBm;                     // The maximum power of the current packet.
        public uint32_t MaxIndex;                         // Index of the maximum power value in the current packet.

        public IQS_Profile_TypeDef IQS_Profile;          // The IQS_ProfileOut function is used to update the IQS configuration information corresponding to the current IQ stream.
        public IQS_StreamInfo_TypeDef IQS_StreamInfo;     // The IQS_Configuration function is used to update the IQS stream format corresponding to the IQ stream.

        public IQS_TriggerInfo_TypeDef IQS_TriggerInfo; // The IQS trigger information corresponding to the current IQ stream is updated through the IQS_GetIQStream function.
        public DeviceInfo_TypeDef DeviceInfo;           // The IQS_GetIQStream function is used to update device information corresponding to the current IQ stream.
        public DeviceState_TypeDef DeviceState;         // The IQS_GetIQStream function is used to update the device status corresponding to the IQ stream.

    };

    /*---------------------------------------------------
    Detector mode structure (Detector)
    -----------------------------------------------------*/
    /*DET configuration structure (basic configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct DET_EZProfile_TypeDef
    {
        public double CenterFreq_Hz;                                // Center frequency.
        public double RefLevel_dBm;                            // R.L.
        public uint32_t DecimateFactor;                           // Decimate factor of time domain data.

        public RxPort_TypeDef RxPort;                               // RF input port.

        public uint32_t BusTimeout_ms;                                // Transmission timeout.

        public DET_TriggerSource_TypeDef TriggerSource;           // Input trigger source.
        public TriggerEdge_TypeDef TriggerEdge;                   // Input trigger edge.

        public TriggerMode_TypeDef TriggerMode;                   // Input trigger mode.	
        public uint64_t TriggerLength;                               // number of sampling points after the input trigger, only available in FixedPoints mode.

        public double TriggerLevel_dBm;                            // Level trigger threshold.
        public double TriggerTimer_Period;                           // Period of timing trigger

        public Detector_TypeDef Detector;                          // Detection.
        public uint16_t DetectRatio;                                  // Detection ratio, the detector detects the power trace, and each original data point is detected as 1 output trace point

        public PreamplifierState_TypeDef Preamplifier;               // Preamplifier action.

        public uint8_t AnalogIFBWGrade;                            //	IF bandwidth grade.

        public ReferenceClockSource_TypeDef ReferenceClockSource; //	Reference clock source.
        public double ReferenceClockFrequency;                       //	Reference clock frequency.

    };

    /* Configuration structure of DET (configuration) */
    [StructLayout(LayoutKind.Sequential)]
    public struct DET_Profile_TypeDef
    {
        public double CenterFreq_Hz;                                      // Center frequency.
        public double RefLevel_dBm;                                     // Reference level.
        public uint32_t DecimateFactor;                                 // Decimate factor of time domain data.

        public RxPort_TypeDef RxPort;                                     // RF input port.

        public uint32_t BusTimeout_ms;                                      // Transmission timeout.

        public DET_TriggerSource_TypeDef TriggerSource;                 // Input trigger source.
        public TriggerEdge_TypeDef TriggerEdge;                          // Input trigger edge.

        public TriggerMode_TypeDef TriggerMode;                         // Input trigger mode.	
        public uint64_t TriggerLength;                                     // The number of sampling points after the input trigger, only available in FixedPoints mode.

        public TriggerOutMode_TypeDef TriggerOutMode;                     // Trigger output mode.
        public TriggerOutPulsePolarity_TypeDef TriggerOutPulsePolarity; // Trigger the output pulse polarity.

        public double TriggerLevel_dBm;                                  // Level trigger threshold.
        public double TriggerLevel_SafeTime;                            // Safety time of level trigger anti-shaking, unit: s.
        public double TriggerDelay;                                    // Trigger delay, unit: s.
        public double PreTriggerTime;                                     // Pre-trigger time, unit: s.

        public TriggerTimerSync_TypeDef TriggerTimerSync;                 // Synchronization options of timer trigger and outer trigger edge.
        public double TriggerTimer_Period;                                 // Period of timed trigger	

        public uint8_t EnableReTrigger;                                 // Enable the device to respond multiple times after capturing a trigger. This function is available only in FixedPoint mode.
        public double ReTrigger_Period;                                 // The interval between multiple responses of the device is also used as the trigger period in the Timer trigger mode (unit: s).
        public uint16_t ReTrigger_Count;                                  // After a trigger, you need to make several responses in addition to the response brought by the trigger.

        public Detector_TypeDef Detector;                              // Detection.
        public uint16_t DetectRatio;                                        // Detection ratio.

        public GainStrategy_TypeDef GainStrategy;                         // Gain strategy.
        public PreamplifierState_TypeDef Preamplifier;                     // Preamplifier action.

        public uint8_t AnalogIFBWGrade;                                //Intermediate frequency bandwidth grade.
        public uint8_t IFGainGrade;                                     //	Intermediate frequency gain grade.

        public uint8_t EnableDebugMode;                                  //	Debug mode. Advanced applications are not recommended. The default value is 0.

        public ReferenceClockSource_TypeDef ReferenceClockSource;         // Reference clock source.
        public double ReferenceClockFrequency;                               //	Reference clock frequency.
        public uint8_t EnableReferenceClockOut;                        // Enable reference clock output.

        public SystemClockSource_TypeDef SystemClockSource;              // System clock source.
        public double ExternalSystemClockFrequency;                         // External system clock frequency: Hz.

        public int8_t Atten;                                              // attenuation.

        public DCCancelerMode_TypeDef DCCancelerMode;                     // Suitable for specific equipment. Dc suppression. 0: disables the DCC. 1: Open, high-pass filter mode (better suppression effect, but will damage the signal in the range of DC to 100 KHZ); 2: Open, manual bias mode (need manual calibration, but not low frequency damage signal).

        public QDCMode_TypeDef QDCMode;                                 // Suitable for specific equipment. IQ amplitude and phase corrector. QDCOff: disables the QDC function. QDCManualMode: Enable and use manual mode; QDCAutoMode: Enables and uses the automatic QDC mode.

        public float QDCIGain;                                              // Suitable for specific equipment. Normalized linear gain I, 1.0 indicates no gain, set range 0.8 to 1.2.
        public float QDCQGain;                                            // Suitable for specific equipment. Normalized linear gain Q, 1.0 indicates no gain, set range 0.8~1.2.
        public float QDCPhaseComp;                                        // Suitable for specific equipment. Normalized phase compensation coefficient, set range -0.2~+0.2.

        public int8_t DCCIOffset;                                         // Suitable for specific equipment. I channel DC offset, LSB.
        public int8_t DCCQOffset;                                         // Suitable for specific equipment. Q channel DC offset, LSB.

        public LOOptimization_TypeDef LOOptimization;                     // LO optimization

    };

    // RBW Filter Type
    public enum RBWFilterType_TypeDef
    {
        RBWFilter_80PercentABW = 0x00,
        RBWFilter_Gaussian_3dB = 0x01,
        RBWFilter_Gaussian_6dB = 0x02,
        RBWFilter_Gaussian_Impulse = 0x03,
        RBWFilter_Gaussian_Noise = 0x04,
        RBWFilter_Flattop = 0x05
    }

    //DET Configuration Structure
    [StructLayout(LayoutKind.Sequential)]
    public struct ZSP_Profile_TypeDef
    {
        public double CenterFreq_Hz;									 // Center frequency
        public double RefLevel_dBm; 									 // Reference Level
        public uint32_t DecimateFactor;									 // Time domain data extraction factor

        public RxPort_TypeDef RxPort;									 // RF Input Port

        public uint32_t BusTimeout_ms;									 // Transmission timeout time

        public DET_TriggerSource_TypeDef TriggerSource;					 // Input trigger source
        public TriggerEdge_TypeDef TriggerEdge;							 // input trigger edge

        public TriggerMode_TypeDef TriggerMode;							 // input trigger mode    
        public uint64_t TriggerLength;									 // Enter the number of samples after the trigger, valid only in FixedPoints mode

        public TriggerOutMode_TypeDef TriggerOutMode;					 // Trigger output mode
        public TriggerOutPulsePolarity_TypeDef TriggerOutPulsePolarity;	 // Trigger output pulse polarity

        public double TriggerLevel_dBm;									 // Level trigger: threshold
        public double TriggerLevel_SafeTime;							 // Level trigger: jitter safe time, unit seconds
        public double TriggerDelay;										 // Level trigger: trigger delay, unit seconds
        public double PreTriggerTime;									 // Level trigger: pre-trigger time, unit seconds

        public TriggerTimerSync_TypeDef TriggerTimerSync;				 // Timed trigger: Synchronization with external trigger edge option
        public double TriggerTimer_Period;                               // Timer trigger: Period                

        public uint8_t EnableReTrigger;									 // Auto ReTrigger: enables the device to respond multiple times after a trigger is captured, only available in FixedPoint mode
        public double ReTrigger_Period;									 // Automatic re-trigger: the time interval between multiple responses, also used as a timer trigger mode trigger period, in seconds
        public uint16_t ReTrigger_Count;								 // automatic retrigger: each time after the original trigger action, the number of automatic retrigger execution

        public Detector_TypeDef Detector;								 // detector
        public uint16_t DetectRatio;									 // Detector ratio, the detector of the power trace detector, each DetectRatio of the original data points detected as 1 output trace point

        public GainStrategy_TypeDef GainStrategy;						 // Gain strategy
        public PreamplifierState_TypeDef Preamplifier;					 // Preamplifier action

        public uint8_t AnalogIFBWGrade;									 // IF bandwidth grade
        public uint8_t IFGainGrade;										 // IF gain grade

        public uint8_t EnableDebugMode;									 // Debug mode, advanced applications do not recommend users to use their own, the default value is 0

        public ReferenceClockSource_TypeDef ReferenceClockSource;		 // Reference clock source
        public double ReferenceClockFrequency;						 	 // Reference Clock Frequency
        public uint8_t EnableReferenceClockOut;						 	 // Enable reference clock out

        public SystemClockSource_TypeDef SystemClockSource;			 	 // System clock source
        public double ExternalSystemClockFrequency; 					 // external system clock frequency, Hz

        public int8_t Atten;                                         // attenuation

        public DCCancelerMode_TypeDef DCCancelerMode;                // Applies to specific devices. DC rejection. 0: DCC off; 1: On, high-pass filter mode (better rejection, but damages signals in the DC to 100kHz range); 2: On, manual bias mode (requires manual calibration, but does not damage signals at low frequencies)

        public QDCMode_TypeDef QDCMode;                              // IQ amplitude-phase corrector. QDCOff: disables the QDC function; QDCManualMode: turns on and uses manual mode; QDCAutoMode: turns on and uses auto QDC mode

        public float QDCIGain; 										 // Device-specific applicable. Normalized linear gain I path, 1.0 means no gain, setting range 0.8~1.2
        public float QDCQGain;										 // Device-specific. Normalized linear gain Q path, 1.0 means no gain, setting range 0.8~1.2
        public float QDCPhaseComp;									 // Device-specific. Normalized phase compensation coefficient, setting range -0.2~+0.2

        public int8_t DCCIOffset; 									 // Device-specific. I-channel DC bias, LSB
        public int8_t DCCQOffset; 									 // device-specific. Q channel DC bias, LSB

        public LOOptimization_TypeDef LOOptimization;                // local oscillation optimization (LOO)

        public double RBW_Hz;                                        // Resolution Bandwidth
        public double VBW_Hz;                                        // Video Bandwidth
        public VBWMode_TypeDef VBWMode;                              // VBW Update Method
        public RBWFilterType_TypeDef RBWFilterType;                  // RBW Filter Type
    }


    /* The stream information structure returned after the DET configuration (returned) */
    [StructLayout(LayoutKind.Sequential)]
    public struct DET_StreamInfo_TypeDef
    {
        public uint64_t PacketCount;      // The total number of data packets corresponding to the current configuration takes effect only in FixedPoints mode.

        public uint64_t StreamSamples;  // In Fixedpoints mode, it represents the total number of sampling points corresponding to the current configuration. In Adaptive mode, the value has no physical significance and is 0.
        public uint64_t StreamDataSize; // In Fixedpoints mode, it indicates the total number of bytes of samples corresponding to the current configuration. In Adaptive mode, the value has no physical significance and is 0.

        public uint32_t PacketSamples;  // Sampling points in packets obtained by each call to DET_GetTrace Indicates the sample points contained in each packet.
        public uint32_t PacketDataSize; // The number of bytes of valid data obtained from each call to DET_GetTrace.
        public double TimeResolution;    // Time-domain point resolution
        public uint32_t GainParameter;   // Gain related parameters, including Space(31 to 24Bit), PreAmplifierState(23 to 16Bit), StartRFBand(15 to 8Bit), StopRFBand(7 to 0Bit).
    };


    /*---------------------------------------------------
    Real-Time Analysis mode structure (RTA)
    -----------------------------------------------------*/
    /*RTA configuration structure (basic configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct RTA_EZProfile_TypeDef
    {
        public double CenterFreq_Hz;                               // Center frequency.
        public double RefLevel_dBm;                              // Reference level.
        public double RBW_Hz;                                      // RBW
        public double VBW_Hz;                                      // VBW

        public RBWMode_TypeDef RBWMode;                           // RBW update mode, mannually, according to the span.
        public VBWMode_TypeDef VBWMode;                          // VBW updated mode,mannually,VBW = RBW,VBW = 0.1*RBW, VBW = 0.01*RBW

        public uint32_t DecimateFactor;                          // Decimate factor of time domain data.

        public Window_TypeDef Window;                              // Window type.

        public SweepTimeMode_TypeDef SweepTimeMode;              // Sweep time mode.
        public double SweepTime;                                  // When the sweep time mode is Manual, the parameter is absolute time. When specified as *N, this parameter is the scan time multiplier
        public Detector_TypeDef Detector;                          // Detector

        public TraceDetectMode_TypeDef TraceDetectMode;           // Trace detection mode.
        public TraceDetector_TypeDef TraceDetector;                // Trace detector.
        public uint32_t TraceDetectRatio;                        // Trace detection ratio.The trace detector detects one output spectrum data point per TraceDetectRatio original spectrum data point.

        public RxPort_TypeDef RxPort;                              // Receiving port setting.

        public uint32_t BusTimeout_ms;                             // Transmission timeout.

        public RTA_TriggerSource_TypeDef TriggerSource;           // Input trigger source.
        public TriggerEdge_TypeDef TriggerEdge;                  // Input trigger edge.

        public TriggerMode_TypeDef TriggerMode;                   // Input trigger mode.
        public double TriggerAcqTime;                               // The sampling time after the input is triggered takes effect only in FixedPoints mode.

        public double TriggerLevel_dBm;                          // Level trigger threshold.
        public double TriggerTimer_Period;                         // Timing trigger period,s.	

        public PreamplifierState_TypeDef Preamplifier;             // Preamplifier action.

        public ReferenceClockSource_TypeDef ReferenceClockSource; //	Reference clock source.
        public double ReferenceClockFrequency;                       //	Reference clock frequency.

    };

    /* Configuration structure of RTA (Configuration) */
    [StructLayout(LayoutKind.Sequential)]
    public struct RTA_Profile_TypeDef
    {
        public double CenterFreq_Hz;                                      // Center frequency.
        public double RefLevel_dBm;                                     // Reference level.
        public double RBW_Hz;                                             // RBW
        public double VBW_Hz;                                            // VBW
        public RBWMode_TypeDef RBWMode;                                  // RBW update mode. Input manually, set automatically according to Span.
        public VBWMode_TypeDef VBWMode;                                  // VBW update mode. Manual input, VBW = RBW, VBW = 0.1*RBW, VBW = 0.01*RBW.

        public uint32_t DecimateFactor;                                 // Decimate factor of time domain data.

        public Window_TypeDef Window;                                     // Window type.

        public SweepTimeMode_TypeDef SweepTimeMode;                     // sweep time mode
        public double SweepTime;                                         // When the sweep time mode is Manual, the parameter is absolute time. When specified as *N, this parameter is the scan time multiplier
        public Detector_TypeDef Detector;                               // Detector

        public TraceDetectMode_TypeDef TraceDetectMode;                 // trace detection mode
        public uint32_t TraceDetectRatio;                               // Trace detection ratio.
        public TraceDetector_TypeDef TraceDetector;                       // Trace detector.

        public RxPort_TypeDef RxPort;                                    // Receiving port setting.

        public uint32_t BusTimeout_ms;                                     // Transmission timeout.

        public RTA_TriggerSource_TypeDef TriggerSource;                 // Input trigger source.
        public TriggerEdge_TypeDef TriggerEdge;                         // Input trigger edge.

        public TriggerMode_TypeDef TriggerMode;                        // Input trigger mode.
        public double TriggerAcqTime;                                      // The sampling time after the input is triggered takes effect only in FixedPoints mode.

        public TriggerOutMode_TypeDef TriggerOutMode;                     // Trigger output mode.
        public TriggerOutPulsePolarity_TypeDef TriggerOutPulsePolarity; // Trigger the output pulse polarity.

        public double TriggerLevel_dBm;                                 // Level trigger threshold.
        public double TriggerLevel_SafeTime;                              // Safety time of level trigger anti-shaking, unit: s.
        public double TriggerDelay;                                     // Trigger delay, unit: s.
        public double PreTriggerTime;                                      // Pre-trigger time, unit: s.

        public TriggerTimerSync_TypeDef TriggerTimerSync;                 // Timer trigger and outer trigger edge synchronization options.
        public double TriggerTimer_Period;                                // Timing trigger period.	

        public uint8_t EnableReTrigger;                                 // Enable the device to respond multiple times after capturing a trigger. This function is available only in FixedPoint mode.
        public double ReTrigger_Period;                                 // The interval between multiple responses of the device is also used as the trigger period in the Timer trigger mode (unit: s).
        public uint16_t ReTrigger_Count;                                 // After a trigger, you need to make several responses in addition to the response brought by the trigger.

        public GainStrategy_TypeDef GainStrategy;                         // Gain strategy.
        public PreamplifierState_TypeDef Preamplifier;                     // Preamplifier action.
        public uint8_t AnalogIFBWGrade;                                 //	Intermediate frequency bandwidth grade.
        public uint8_t IFGainGrade;                                     //	Intermediate frequency gain grade.

        public uint8_t EnableDebugMode;                                 //	Debug mode. Advanced applications are not recommended. The default value is 0.

        public ReferenceClockSource_TypeDef ReferenceClockSource;         //	Reference clock source.
        public double ReferenceClockFrequency;                             //	Reference clock frequency.
        public uint8_t EnableReferenceClockOut;                         // Enable reference clock output.

        public SystemClockSource_TypeDef SystemClockSource;              // System clock source.
        public double ExternalSystemClockFrequency;                         // External system clock frequency: Hz.

        public int8_t Atten;                                            // attenuation.

        public DCCancelerMode_TypeDef DCCancelerMode;                    // Suitable for specific equipment. Dc suppression. 0: disables the DCC. 1: Open, high-pass filter mode (better suppression effect, but will damage the signal in the range of DC to 100 KHZ); 2: Open, manual bias mode (need manual calibration, but not low frequency damage signal).

        public QDCMode_TypeDef QDCMode;                                // Suitable for specific equipment. IQ amplitude and phase corrector. QDCOff: disables the QDC function. QDCManualMode: Enable and use manual mode; QDCAutoMode: Enables and uses the automatic QDC mode.

        public float QDCIGain;                                             // Suitable for specific equipment. Normalized linear gain I, 1.0 indicates no gain, set range 0.8 to 1.2.
        public float QDCQGain;                                            // Suitable for specific equipment. Normalized linear gain Q, 1.0 indicates no gain, set range 0.8~1.2.
        public float QDCPhaseComp;                                        // Suitable for specific equipment. Normalized phase compensation coefficient, set range -0.2~+0.2.

        public int8_t DCCIOffset;                                         // Suitable for specific equipment. I channel DC offset, LSB.
        public int8_t DCCQOffset;                                         // Suitable for specific equipment. Q channel DC offset, LSB.

        public LOOptimization_TypeDef LOOptimization;                     // LO optimization.

    };

    /* Package information structure returned after RTA configuration (Returned) */
    [StructLayout(LayoutKind.Sequential)]
    public struct RTA_FrameInfo_TypeDef
    {
        public double StartFrequency_Hz;     // The start frequency of the spectrum.	
        public double StopFrequency_Hz;     // The stop frequency of the spectrum.
        public double POI;                     // The shortest duration of the signal with 100% probability of interception, unit: s.

        public double TraceTimestampStep;   // Timestamp step of each Trace in each packet of data. (package overall timestamp in TriggerInfo SysTimerCountOfFirstDataPoint).
        public double TimeResolution;          // The sampling time of each time domain data which is also the resolution of the timestamp.
        public double PacketAcqTime;          // Data collection time of each packet.

        public uint32_t PacketCount;        // The total number of data packets corresponding to the current configuration takes effect only in FixedPoints mode.
        public uint32_t PacketFrame;        // The number of valid frames per packet.
        public uint32_t FFTSize;            // The number of FFTS per frame.
        public uint32_t FrameWidth;         // The number of points after FFT frame interception is also the number of points of each Trace in the packet, which can be used as the number of points on the X-axis of the probability density graph (width).
        public uint32_t FrameHeight;        // The spectrum amplitude range corresponding to the FFT frame can be used as the number of Y-axis points (height) of the probability density map.

        public uint32_t PacketSamplePoints; // Number of collection points corresponding to each packet of data.
        public uint32_t PacketValidPoints;     // The number of valid data points in the frequency domain contained in each packet.

        public uint32_t MaxDensityValue;     // Upper limit of individual site element value of probability density bitmap.
        public uint32_t GainParameter;         // Include Space(31~24Bit),PreAmplifierState(23~16Bit),StartRFBand(15~8Bit),StopRFBand(7~0Bit)
    };

    /* The drawing information structure returned by RTA after obtaining it */
    [StructLayout(LayoutKind.Sequential)]
    public struct RTA_PlotInfo_TypeDef
    {
        public float ScaleTodBm;              // Compression from linear power to logarithmic power. The absolute power of Trace is equal to Trace[] * ScaleTodBm + OffsetTodBm(the absolute power axis of Bitmap is the same below).
        public float OffsetTodBm;            // The shift of relative power into absolute power. The absolute power axis range of bitmap (Y-axis) is equal to FrameHeigh * ScaleTodBm + OffsetTodBm(Trace physical power ditto above).
        public uint64_t SpectrumBitmapIndex; // The number of times a probability density map is obtained, which can be used as an index when plotting.
    };


    /*---------------------------------------------------
    Multi-profile schema Association (MPS)
    -----------------------------------------------------*/
    [StructLayout(LayoutKind.Sequential)]
    public struct MPSDataInfo_TypeDef
    {
        public uint8_t Analysis;
        public uint16_t ProfileIndex;
        public float IQS_ScaleToV;
        public float DET_ScaleToV;
        public RTA_PlotInfo_TypeDef RTA_PlotInfo;

    };

    /*---------------------------------------------------
    Calibration correlation (CAL)
    -----------------------------------------------------*/

    /*CAL's configuration struct (configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct CAL_QDCOptParam_TypeDef
    {
        public double RFCFreq_Hz;               // Device RF center frequency: Hz.
        public double BBCFreq_Hz;               // Frequency point of calibration signal selected, Hz.
        public float Temperature;           // Equipment temperature.

        public float RefLevel_dBm;           // Device reference level, dBm.
        public float CalSignalLevel_dBm;       // Calibration signal power, dBm.

        public float QDCIGain;               // IGain optimization value obtained.
        public float QDCQGain;               // QGain optimization value obtained.
        public float QDCPhaseComp;           // The PhaseComp optimization value is obtained.
        public float EstimatedOptRejection; // The expected sideband suppression value obtained after the optimization value is substituted.

    };

    /*---------------------------------------------------
    Source functional struct (Signal)
    -----------------------------------------------------*/

    /* Manual gain control structure in emission mode. The secondary parameter takes effect when the emission source works in SIG_ManualGainCtrl mode */
    [StructLayout(LayoutKind.Sequential)]
    public struct TxManualGain_TypeDef
    {
        public uint8_t TxPreDSA;      // Preattenuator value; Input range: 0-127; Step 1.
        public uint8_t TxPostDSA;     // Rear attenuator value; Input range: 0-127; Step 1.
        public uint8_t TxAuxDSA;      // Auxiliary attenuator value; Input range: 0-127; Step 1.
        public uint8_t PreAmplifier;  // Preamplifier status: 0: off; 1: Turn on.
        public uint8_t PostAmplifier; // Post-amplifier status: 0: off; 1: Turn on.
    };

    /*The playback mode of the signal generator IQStream*/
    public enum SIG_IQStreamPlayMode_TypeDef
    {
        SIG_IQStreamPlayMode_Null = 0x00,               // Real-time transmission and playback mode.
        SIG_IQStreamPlayMode_RealTime = 0x01,           // Real-time transmission and playback mode.
        SIG_IQStreamPlayMode_SinglePlay = 0x02,         // Single play.
        SIG_IQStreamPlayMode_ContinousPlay = 0x04,      // Continuous play.
        SIG_IQStreamPlayMode_MultiPlay = 0x08,          // Multiple play.
        SIG_IQStreamPlayMode_Simulation = 0x10          // Simulation signal.

    };

    /*Signal generator IQStream playback trigger source*/
    public enum SIG_IQStreamPlayTrigger_TypeDef
    {
        SIG_IQStreamPlayTrigger_Null = 0x00,        // No trigger; Wait for rematch to another trigger type.
        SIG_IQStreamPlayTrigger_Internal = 0x01,    // Internal trigger.
        SIG_IQStreamPlayTrigger_External = 0x02,    // External trigger.
        SIG_IQStreamPlayTrigger_Timer = 0x04,       // Timing trigger.
        SIG_IQStreamPlayTrigger_RFBoard = 0x08,     // Timing trigger.
        SIG_IQStreamPlayTrigger_Bus = 0x10          // Bus trigger.

    };

    /*SIG mode user interface definition(configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct SIG_Profile_TypeDef
    {
        public double CenterFreq_Hz;                                    // The current center frequency (unit: Hz) takes effect when the signal generator works in SIG_Fixed mode; Input range 1M-1GHz; Step 1Hz.
        public double StartFreq_Hz;                                     // The initial frequency in frequency sweep mode, expressed in Hz, takes effect when the signal generator is operating in SIG_FreqSweep_* mode. Input range 1M-1GHz; Step 1Hz.
        public double StopFreq_Hz;                                      // The termination frequency in frequency sweep mode, expressed in Hz, takes effect when the signal generator is operating in SIG_FreqSweep_* mode. Input range 1M-1GHz; Step 1Hz.
        public double StepFreq_Hz;                                      // Step frequency in frequency sweep mode, expressed in Hz, takes effect when the signal generator works in SIG_FreqSweep_* mode. Input range 1M-1GHz; Step 1Hz.
        public double CurrentLevel_dBm;                                 // The current power, in dBm, takes effect when the signal generator is working in SIG_Fixed mode; Input range -127 to -5dBm; Step by 0.25dB.
        public double StartLevel_dBm;                                   // The unit is Hz. This parameter takes effect when the signal generator works in SIG_PowerSweep_* mode. Input range -127 to -5dBm; Step by 0.25dB.
        public double StopLevel_dBm;                                    // The termination power in power sweep mode, expressed in Hz, takes effect when the signal generator works in SIG_PowerSweep_* mode. Input range -127 to -5dBm; Step by 0.25dB.
        public double StepLevel_dBm;                                    // Step power in power sweep mode, expressed in Hz, takes effect when the signal generator works in SIG_PowerSweep_* mode. Input range -127 to -5dBm; Step by 0.25dB.
        public double DwellTime_ms;                                     // Sweep dwell time, effective when the generator works in the *Sweep* mode; Input range 0-1000000; Step 1.
        public double DACSampleRate;                                    // Specifies the sampling rate for the DAC.
        public double InterpolationFactor;                              // signal generator baseband interpolation multiple, this parameter determines the baseband bandwidth of the signal generator, input range :1-1024; Input is an even number except for 1.
        public double ReferenceClockFrequency;                          // Reference input frequency in signal generator mode; Currently, this parameter only supports 10MHz reference frequency input. Tuning this parameter can compensate the frequency bias of internal or external reference frequency.
        public ReferenceClockSource_TypeDef ReferenceClockSource;       // Reference frequency input source; 0: internal 1: external.
        public SIG_OperationMode_TypeDef OperationMode;                 // Working mode of signal generator.
        public SIG_SweepSource_TypeDef SweepSource;                     // Signal generator  Scan source.

        public SIG_IQStreamPlayMode_TypeDef SIG_IQStreamPlayMode;       // Play mode of signal generator IQStream.
        public SIG_IQStreamPlayTrigger_TypeDef SIG_IQStreamPlayTrigger; // Trigger of plays of the signal generator IQStream.
        public uint32_t SIG_IQStreamPlayLength;                         // Length of plays of the signal generator IQStream.
        public uint32_t SIG_IQStreamPlayCounts;                         // Number of plays of the signal generator IQStream.
        public uint32_t SIG_IQStreamPlayPrevDelay;                      // The preview delay of signal generator IQStream playback is effective for each playback.
        public uint32_t SIG_IQStreamPlayPostDelay;                      // The post delay of signal generator IQStream playback is effective for each playback.

        public uint32_t SIG_IQStreamDownloadStartAddress;               // The signal generator IQStream is downloaded to the starting memory address in the onboard memory.
        public uint32_t SIG_IQStreamDownloadStopAddress;                // The signal generator IQStream is downloaded to the stopping  memory address in the onboard memory.

        public uint32_t SIG_IQStreamPlayStartAddress;                   // The signal generator reads the starting memory address of IQStream from the onboard memory.
        public uint32_t SIG_IQStreamPlayStopAddress;                    // The signal source reads the stopping memory address of IQStream from the onboard memory.

        public int16_t SIG_IQStreamIdle_DC_I;                           // Silent DC value of channel I signal in non-play mode.
        public int16_t SIG_IQStreamIdle_DC_Q;                           // Silent DC value of the Q channel signal in non-play mode.

        public int16_t SIG_IQStream_DC_Offset_I;                        // The DC offset of the I signal ranges from -32768 to +32767.
        public int16_t SIG_IQStream_DC_Offset_Q;                        // The DC offset of the Q signal ranges from -32768 to +32767.

        public double SIG_IQStream_Gain_I;                              // Linear gain of channel I signals: The gain ranges from -256 to 255.
        public double SIG_IQStream_Gain_Q;                              // Linear gain of the Q-channel signal: The gain ranges from -256 to 255.

        public double SIG_IQStreamSimFrequency;                         // When SIG_IQStreamPlayMode=Simualtion mode, specify the baseband frequency of emulation.

        public TxPort_TypeDef TxPort;                                   // signal generator output port.
        public TxTriggerInSource_TypeDef TxTriggerInSource;             // The signal generator triggers the input source.
        public TxTriggerInMode_TypeDef TxTriggerInMode;                 // signal generator output input mode.
        public TxTriggerOutMode_TypeDef TxTriggerOutMode;               // The signal generator triggers the output mode.
        public TxAnalogIQSource_TypeDef TxAnalogIQSource;               // signal generator simulates IQ input source.
        public TxDigitalIQSource_TypeDef TxDigitalIQSource;             // signal generator Digital QI input source.
        public TxTransferReset_TypeDef TransferResetCmd;                // signal generator baseband transmission reset status.
        public TxPackingCmd_TypeDef TxPackingCmd;                       // Indicates the packet mode status of the signal source.
        public TxManualGain_TypeDef TxManualGain;                       // Source manual gain control.
    };

    /* Source Mode Return information (to be updated)(Return) */
    [StructLayout(LayoutKind.Sequential)]
    public struct SIG_Info_TypeDef
    {
        public uint32_t SweepPoints; // scan points
    };

    /*---------------------------------------------------
    Analog Signal Generator(subfunction of SIG)
    -----------------------------------------------------*/

    /*working mode of ASG*/
    public enum ASG_Mode_TypeDef
    {
        ASG_Mute = 0x00,            // mute.
        ASG_FixedPoint = 0x01,      // Fixed point.
        ASG_FrequencySweep = 0x02,  // Frequency sweep.
        ASG_PowerSweep = 0x03       // Power sweep.

    };

    /*Input trigger source of ASG*/
    public enum ASG_TriggerSource_TypeDef
    {
        ASG_TriggerIn_FreeRun = 0x00,  // Free running.
        ASG_TriggerIn_External = 0x01, // External trigger.
        ASG_TriggerIn_Bus = 0x02       // Timer trigger.

    };

    /*Input trigger mode of ASG*/
    public enum ASG_TriggerInMode_TypeDef
    {
        ASG_TriggerInMode_Null = 0x00,        // Free running.
        ASG_TriggerInMode_SinglePoint = 0x01, // Single point trigger (a configuration of frequency or power that is triggered once for a single time).
        ASG_TriggerInMode_SingleSweep = 0x02, // Single scan trigger (trigger one cycle scan at a time).
        ASG_TriggerInMode_Continous = 0x03    // Continuous scan trigger (trigger one continuous operation).
    };

    /*Trigger output port of ASG*/
    public enum ASG_TriggerOutPort_TypeDef
    {
        ASG_TriggerOut_Null = 0x00,         // external
        ASG_TriggerOut_External = 0x01,     // external
        ASG_TriggerOut_Bus = 0x02           // internal
    };

    /*Trigger output mode of ASG*/
    public enum ASG_TriggerOutMode_TypeDef
    {
        ASG_TriggerOutMode_Null = 0x00,        // Free running.
        ASG_TriggerOutMode_SinglePoint = 0x01, // Single point trigger (a configuration of frequency or power that is triggered once for a single time).
        ASG_TriggerOutMode_SingleSweep = 0x02  // Single scan trigger (trigger one cycle scan at a time).
    };

    /*RF port of ASG*/
    public enum ASG_Port_TypeDef
    {
        ASG_Port_External = 0x00,   // external
        ASG_Port_Internal = 0x01,   // internal
        ASG_Port_ANT = 0x02,        // ANT Port (TRx series only).
        ASG_Port_TR = 0x03,         // TR Port (TRx series only).
        ASG_Port_SWR = 0x04,        // SWR Port (TRx series only).
        ASG_Port_INT = 0x05         // INT Port (TRx series only).

    };

    /*Configuration struck of ASG(configuration )*/
    [StructLayout(LayoutKind.Sequential)]
    public struct ASG_Profile_TypeDef
    {
        public double CenterFreq_Hz;                                // The current center frequency (unit: Hz) takes effect when the signal generator works in SIG_Fixed mode; Input range 1M-1GHz; Step 1Hz.
        public double Level_dBm;                                    // The current power, in dBm, takes effect when the signal generator is working in SIG_Fixed mode; Input range -127 to -5dBm; Step by 0.25dB.

        public double StartFreq_Hz;                                 // The initial frequency in frequency sweep mode, expressed in Hz, takes effect when the signal generator is operating in SIG_FreqSweep_* mode. Input range 1M-1GHz; Step 1Hz.
        public double StopFreq_Hz;                                  // The termination frequency in frequency sweep mode, expressed in Hz, takes effect when the signal generator is operating in SIG_FreqSweep_* mode. Input range 1M-1GHz; Step 1Hz.
        public double StepFreq_Hz;                                  // Step frequency in frequency sweep mode, expressed in Hz, takes effect when the signal generator works in SIG_FreqSweep_* mode. Input range 1M-1GHz; Step 1Hz.

        public double StartLevel_dBm;                               // Start power in power sweep mode (unit: Hz).
        public double StopLevel_dBm;                                // End power in power sweep mode (unit: Hz).
        public double StepLevel_dBm;                                // Power Step power in sweep mode (unit: Hz).

        public double DwellTime_s;                                  // In frequency Sweep mode or power sweep mode, the unit is s. When the triggering mode is BUS, the sweep dwell time, the unit is s, takes effect when the signal generator works in the *Sweep* mode. Input range 0-1000000; Step 1.

        public double ReferenceClockFrequency;                      // Specified reference frequency: both internal and external references are effective.
        public ReferenceClockSource_TypeDef ReferenceClockSource;   // Select the input source of the reference clock: internal reference or external reference.

        public ASG_Port_TypeDef Port;                               // signal generator output port.

        public ASG_Mode_TypeDef Mode;                               // Off, dot frequency, frequency scan (external trigger, synchronize to receive), power scan (external trigger, synchronize to receive).

        public ASG_TriggerSource_TypeDef TriggerSource;             // signal generator trigger input mode.
        public ASG_TriggerInMode_TypeDef TriggerInMode;             // Trigger mode of the signal source.
        public ASG_TriggerOutMode_TypeDef TriggerOutMode;           // Trigger mode of signal source.

    };

    /* Analog Source Mode Return Information (to be updated)(Return) */
    [StructLayout(LayoutKind.Sequential)]
    public struct ASG_Info_TypeDef
    {
        public uint32_t SweepPoints; // Scan Points
    };

    /*---------------------------------------------------
    Vector Signal Generator(subfunction of SIG)
    -----------------------------------------------------*/

    /*working mode of VSG*/
    public enum VSG_Mode_TypeDef
    {
        VSG_Mode_Null = 0x00,           // Mute
        VSG_Mode_RealTime = 0x01,       // Real time mode.
        VSG_Mode_PlayFromRAM = 0x02,    // Reads data from memory and plays it back.
        VSG_Mode_SimulatedCW = 0x04,    // Continuous wave patterns generated by internal simulation.
        VSG_Mode_Advanced = 0x08        // Advanced Mode (not yet available).

    };

    /*Trigger inter source of VSG*/
    public enum VSG_TriggerSource_TypeDef
    {
        VSG_TriggerSource_Null = 0x00,          // No trigger; Wait for rematch to another trigger type.
        VSG_TriggerSource_Internal = 0x01,      // Internal trigger.
        VSG_TriggerSource_External = 0x02,      // External trigger.
        VSG_TriggerSource_Timer = 0x04,         // Timing trigger.
        VSG_TriggerSource_RFBoard = 0x08,       // RF Board trigger.
        VSG_TriggerSource_Bus = 0x10            // Bus trigger.

    };

    /*RF output port of VSG*/
    public enum VSG_Port_TypeDef
    {
        VSG_Port_External = 0x00,   // External port.
        VSG_Port_Internal = 0x01,   // Internal port.
        VSG_Port_ANT = 0x02,        // ANT Port (TRx series only).
        VSG_Port_TR = 0x03,         // TR Port (TRx series only).
        VSG_Port_SWR = 0x04,        // SWR Port (TRx series only).
        VSG_Port_INT = 0x05         // INT Port (TRx series only).

    };

    /*configuration struck of VSG (configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct VSG_Profile_TypeDef
    {
        public double CenterFrequency;                                  // Center frequency of the VSG.
        public double Outlevel;                                         // Power output of the VSG.
        public double DACSampleRate;                                    // Sample rate of the VSG.
        public double InterpolationFactor;                              // Interpolation factor of the VSG.
        public uint32_t Triggerlength;                                  // The length of the IQStream playback of the VSG.
        public int32_t TriggerCount;                                    // The number of times the IQStream playback of the VSG.
        public uint32_t TriggerDelay_Prev;                              // The preview delay of IQStream playback is effective for each playback.
        public uint32_t TriggerDelay_Post;                              // The post delay of signal generator IQStream playback is effective for each playback.
        public uint32_t PlayStartAddress;                               // The VSG IQStream is downloaded to the starting memory address in the onboard memory.
        public uint32_t PlayStopAddress;                                // The VSG IQStream is downloaded to the stopping memory address in the onboard memory.
        public double ReferenceClockFrequency;                          // Reference input frequency in signal source mode; Currently, this parameter only supports 10MHz reference frequency input. Tuning this parameter can compensate the frequency bias of internal or external reference frequency.
        public ReferenceClockSource_TypeDef ReferenceClockSource;       // The reference clock source of the VSG.
        public VSG_Mode_TypeDef Mode;                                   // The working mode of the VSG.
        public VSG_Port_TypeDef Port;                                   // The RF output port of the VSG.
        public VSG_TriggerSource_TypeDef TriggerSource;                 // The trigger source of the VSG.
    };

    /*VSG mode advanced parameters (configuration), only for VSG_Mode_Advanced mode*/
    [StructLayout(LayoutKind.Sequential)]
    public struct VSG_PrivateParam_TypeDef
    {
        public double StartFreq_Hz;                        // The start frequency in frequency sweep mode, expressed in Hz, takes effect when the signal source is operating in SIG_FreqSweep_* mode. Input range 1M-1GHz; Step 1Hz.
        public double StopFreq_Hz;                         // The stop frequency in frequency sweep mode, expressed in Hz, takes effect when the signal source is operating in SIG_FreqSweep_* mode. Input range 1M-1GHz; Step 1Hz.
        public double StepFreq_Hz;                         // The step frequency in frequency sweep mode, expressed in Hz, takes effect when the signal source is operating in SIG_FreqSweep_* mode. Input range 1M-1GHz; Step 1Hz.
        public double StartLevel_dBm;                      // The starting power in power sweep mode, expressed in Hz, takes effect when the signal source works in SIG_PowerSweep_* mode. Input range -127 to -5dBm; Step by 0.25dB.
        public double StopLevel_dBm;                       // The stopping power in power sweep mode, expressed in Hz, takes effect when the signal source works in SIG_PowerSweep_* mode. Input range -127 to -5dBm; Step by 0.25dB.
        public double StepLevel_dBm;                       // The step power in power sweep mode, expressed in Hz, takes effect when the signal source works in SIG_PowerSweep_* mode. Input range -127 to -5dBm; Step by 0.25dB.
        public double DwellTime_ms;                        // Sweep dwell time, effective when the source works in the *Sweep* mode; Input range 0-1000000; Step 1.

        public SIG_OperationMode_TypeDef OperationMode;         // Working mode of signal source.
        public SIG_SweepSource_TypeDef SweepSource;             // Signal source Scan source.

        public int16_t SIG_IQStreamIdle_DC_I;                   // Silent DC value of channel I signal in non-play mode.
        public int16_t SIG_IQStreamIdle_DC_Q;                   // Silent DC value of the Q channel signal in non-play mode.

        public int16_t SIG_IQStream_DC_Offset_I;                // The DC offset of the I signal ranges from -32768 to +32767.
        public int16_t SIG_IQStream_DC_Offset_Q;                // The DC offset of the Q signal ranges from -32768 to +32767.

        public double SIG_IQStream_Gain_I;                      // Linear gain of channel I signals: The gain ranges from -256 to 255.
        public double SIG_IQStream_Gain_Q;                      // Linear gain of the Q-channel signal: The gain ranges from -256 to 255.
        public double SIG_IQStreamSimFrequency;                 // When SIG_IQStreamPlayMode=Simualtion mode, specify the baseband frequency of emulation.

        public TxTriggerInSource_TypeDef TxTriggerInSource;     // The signal source triggers the input source.
        public TxTriggerInMode_TypeDef TxTriggerInMode;         // Signal source output input mode.
        public TxTriggerOutMode_TypeDef TxTriggerOutMode;       // The signal source triggers the output mode.
        public TxAnalogIQSource_TypeDef TxAnalogIQSource;       // Signal source simulates IQ input source.
        public TxDigitalIQSource_TypeDef TxDigitalIQSource;     // Signal source Digital QI input source.
        public TxTransferReset_TypeDef TransferResetCmd;        // Signal source baseband transmission reset status.
        public TxPackingCmd_TypeDef TxPackingCmd;               // Indicates the packet mode status of the signal source.
        public TxManualGain_TypeDef TxManualGain;               // Source manual gain control.
    };

    /*VSG parameters return information (return)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct VSG_Info_TypeDef
    {
        public uint32_t PlayAddressRange; //Scan points
    };

    /*VSG file format*/
    public enum VSG_FileFormat_TypeDef
    {
        VSG_FileFormat_wav = 0x00, // wav format.
        VSG_FileFormat_dat = 0x01  // dat format.

    };

    /*VSG information returned from playing the file (return)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct VSG_FileInfo_TypeDef
    {
        public uint32_t FileSize_Byte;          // File size (bytes).
        public uint32_t IQStreamLength;         // The length of IQstream(sampling points).
        public uint32_t PlayStartAddress;       // The starting address of the on-board memory.
        public uint32_t PlayStopAddress;        // The stopping address of the on-board memory.
        public uint32_t NextFileStartAddress;   // Indicates the starting memory address at which the next file or segment can be downloaded and played.
        public uint32_t PlayAddressRange;       // The address range for the file to play.	
        public uint32_t AddressOverRangeFlag;   // This flag is set when the write address exceeds the maximum address.
        public uint32_t File_Format;            // File format.
        public double FileSampleRate;           // The sampling rate of the IQStream in the file.

    };

    /*---------------------------------------------------
Vector Network Analyzer mode struck(VNA)
-----------------------------------------------------*/

    /* Vector Network Analyzer Mode User Interface Definition (Configuration) */
    [StructLayout(LayoutKind.Sequential)]
    public struct VNA_Profile_TypeDef
    {
        public double CenterFreq_Hz;                                // When set to power scan or fixed mode, this parameter is the center frequency; Input range 1M-1GHz; Step 1Hz.
        public double StartFreq_Hz;                                 // Initial frequency in S11/S21 scanning mode, expressed in Hz; Input range 1M-1GHz; Step 1Hz.
        public double StopFreq_Hz;                                  // Termination frequency in S11/S21 scanning mode, expressed in Hz; Input range 1M-1GHz; Step 1Hz.
        public double StepFreq_Hz;                                  // Step frequency in S11/S21 scanning mode, expressed in Hz; Input range 1M-1GHz; Step 1Hz.
        public double TxOutLevel_dBm;                               // This parameter specifies the current transmitting power of the transmitter; Input range -127 to -5dBm; Step by 0.25dB.
        public double TxStartOutLevel_dBm;                          // This parameter specifies the initial transmitting power of the transmitter; Input range -127 to -5dBm; Step by 0.25dB.
        public double TxStopOutLevel_dBm;                           // This parameter specifies the terminal transmission power of the transmitter; Input range -127 to -5dBm; Step by 0.25dB.
        public double TxStepOutLevel_dBm;                           // This parameter specifies the step transmission power of the transmitter; Input range -127 to -5dBm; Step by 0.25dB.
        public double RxRefLevel_dBm;                               // This parameter specifies the offset of the receiver's reference level or reference level with respect to transmitted power or, when synchronized with transmitted power, with respect to transmitted power; Input range -127--5dBm; Step by 0.25dB.
        public double RBW;                                          // Analysis bandwidth of the specified receiver; Input range 1Hz-1MHz (to be determined); Step 1Hz.
        public double DwellTime_us;                                 // Specifies the dwell time. This parameter is applicable when the transmission delay of the device under test is large. Input range 0-1000000; Step 1.
        public double ReferenceClockFrequency;                      // Reference input frequency in signal source mode; Currently, this parameter only supports 10MHz reference frequency input. Tuning this parameter can compensate the frequency bias of internal or external reference frequency.
        public ReferenceClockSource_TypeDef ReferenceClockSource;   // Reference frequency input source; 0: internal 1: external.
        public VNA_OperationMode_TypeDef OperationMode;             // Working mode of network analyzer
        public RxLevelMode_typedef RxLevelMode;                     // When operating in power sweep mode, this parameter specifies the operating state of the receiver's reference level: fixed and in sync with the transmitter.
        public TRxTriggerInSource_typedef TriggerInSource;          // Trigger input source.
        public TRxTriggerInMode_typedef TriggerInMode;              // Trigger input mode.
        public TRxTriggerOutMode_typedef TriggerOutMode;            // Trigger output mode.
    };

    /*Return information for vector network analysis mode (to be updated)(Returned)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct VNA_Info_TypeDef
    {
        public double StartFreq_Hz; // Initial frequency in S11/S21 scanning mode, expressed in Hz; Input range 1M-1GHz; Step 1Hz.
        public double StopFreq_Hz;  // Termination frequency in S11/S21 scanning mode, expressed in Hz; Input range 1M-1GHz; Step 1Hz.
        public double StepFreq_Hz;  // Step frequency in S11/S21 scanning mode, expressed in Hz; Input range 1M-1GHz; Step 1Hz.
        public int SweepPoints;      // Scan number.	
        public int AnalysisPoints;   // Analysis points.
    };

    /*---------------------------------------------------
Analog modulation and demodulation Structure (Analog)
    -----------------------------------------------------*/

    /*Signal source FM modulation signal interface definition (configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct SIG_FM_Profile_TypeDef
    {
        public double SignalFrequency;                    // FM modulation signal frequency: Input range: not yet determined.
        public double ModDeviation;                       // FM modulation frequency offset: Input range: not yet determined.
        public double SampleRate;                         // Sampling rate: This sampling rate = DAC original sampling rate/interpolation multiple.
        public short Amplitude;                           // Signal amplitude; Input range: 0-8191, step 1.
        public int resetflag;                             // Reset state. After reconfiguration, the first run should be set to 1 and then to 0.
        public AnalogSignalType_TypeDef AnalogSignalType; // Analog signal type.
    };

    // AM Modulation Parameter Structure
    [StructLayout(LayoutKind.Sequential)]
    public struct AM_DemodParam_TypeDef
    {
        public double ModRate; // AM modulation rate
        public double ModDepth; // AM modulation depth or modulation index
        public double ModRate_Avg; // AM modulation rate after multiple averages
        public double ModDepth_Avg; // AM modulation depth or modulation index after multiple averages
    }

    // FM Modulation Parameter Structure
    [StructLayout(LayoutKind.Sequential)]
    public struct FM_DemodParam_TypeDef
    {
        public double ModRate; // FM modulating signal frequency
        public double ModDeviation; // FM modulation frequency offset
        public double CarrierFreqOffset; // Carrier offset
        public double ModRate_Avg; // FM modulated signal frequency after multiple averaging
        public double ModDeviation_Avg; // FM modulation frequency offset after multiple averages
        public double CarrierFreqOffset_Avg; // Carrier offset after multiple averages
    }

    /*Signal source AM modulation signal interface definition (configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct SIG_AM_Profile_TypeDef
    {
        public double SignalFrequency;                                  // AM modulation signal frequency: Input range: not yet determined.
        public double ModIndex;                                         // AM modulation index: Input range: not yet determined.
        public double SampleRate;                                       // Sampling rate: This sampling rate = DAC original sampling rate/interpolation multiple.
        public short Amplitude;                                         // Signal amplitude; 0-8191.
        public int resetflag;                                           // Reset state. After reconfiguration, the first run should be set to 1 and then to 0.
        public AnalogSignalType_TypeDef AnalogSignalType;               // Analog signal type.
        public AMModType_TypeDef AMModType;                             // AM modulation type.
        public AM_CarrierSuppression_Typedef AMCarrierSuppression;      // AM modulated carrier suppression.
    };

    [StructLayout(LayoutKind.Sequential)]
    public struct SIG_Pulse_Profile_TypeDef
    {
        public double PulseRepetitionFrequency;      // Pulse repetition rate (PRF).
        public double SampleRate;                        // Sampling rate: This sampling rate = DAC original sampling rate/interpolation multiple.
        public double PulseDepth;                        // Pulse modulation depth, that is, on-off power ratio, unit is dB; And it must be positive.
        public double PulseDuty;                         // The duty cycle of the pulse:0-1.
        public short Amplitude;                          // Signal amplitude:0-8191.
        public int resetflag;                   		 // Reset state. After reconfiguration, the first run should be set to 1 and then to 0.
    };

    /*---------------------------------------------------
DSP auxiliary function architecture (DSP)
    -----------------------------------------------------*/

    /*Configuration structure for the internal clock calibration source*/
    public enum ClkCalibrationSource_TypeDef
    {
        CalibrateByExternal = 0x00, // Clock calibration via Ext trigger.
        CalibrateByGNSS1PPS = 0x01  // Clock calibration was carried out by GNSS-1PPS.
    };

    /*Configuration structure for Fast Fourier Transform mode (Configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct DSP_FFT_TypeDef
    {
        public uint32_t FFTSize;                        // FFT analysis points.
        public uint32_t SamplePts;                      // Effective sampling number.
        public Window_TypeDef WindowType;               // Window type.

        public TraceDetector_TypeDef TraceDetector;     // Video detection mode.
        public uint32_t DetectionRatio;                 // Trace detection ratio.
        public float Intercept;                         // Output spectrum interception, such as Intercept = 0.8, means that 80% of the spectrum results are output.

        public bool Calibration;                        // Calibrate or not.
    };

    /*The parameters needed to generate the filter coefficient (configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct Filter_TypeDef
    {
        public int n;            // Number of filter taps(n > 0 ).
        public float fc;         // Cut-off frequency ( Cut-off frequency/sampling rate  0 < fc < 0.5 ).
        public float As;         // Stopband attenuation() As > 0 , [dB]).
        public float mu;         // Fractional sampling offset( -0.5 < mu < 0.5 ).
        public uint32_t Samples; // Sampling number(Samples > 0).

    };

    /*The parameters required to generate the sine wave (configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct Sin_TypeDef
    {
        public double Frequency;  // The frequency of the sine wave.
        public float Amplitude;   // The magnitude of the sine wave.
        public float Phase;       // The phase of the sine wave.
        public double SampleRate; // The sampling rate of the sine wave.
        public uint32_t Samples;  // The number of sinewave samples.
    };

    /*Configuration structure for digital down conversion mode (Configuration)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct DSP_DDC_TypeDef
    {
        public double DDCOffsetFrequency; // The frequency offset of complex mixing.
        public double SampleRate;         // Sample rate of complex mixing.
        public float DecimateFactor;      // Resampling extraction multiple, range :1 ~ 2^16.
        public uint64_t SamplePoints;     // Sampling number.
    };

    /*IP3 test structure*/
    [StructLayout(LayoutKind.Sequential)]
    public struct TraceAnalysisResult_IP3_TypeDef
    {
        public double LowToneFreq;          // Bass frequency, in units depending on the data source.
        public double HighToneFreq;         // High-pitched signal frequency, in units depending on the data source.
        public double LowIM3PFreq;          // Low frequency intermodulation frequency, unit dependent on data source.
        public double HighIM3PFreq;         // High frequency intermodulation frequency, unit depending on the data source.

        public float LowTonePower_dBm;      // Bass power, dBm.
        public float HighTonePower_dBm;     // Treble power, dBm.
        public float TonePowerDiff_dB;      // Bass power - treble power.
        public float LowIM3P_dBc;           // LowIM3P_dBc = max(LowTonePower_dBm, HighTonePower_dBm) - LowTonePower_dBm, The intensity of the low frequency crossmodulation product relative to the strongest main signal.
        public float HighIM3P_dBc;          // HighIM3P_dBc = max(LowTonePower_dBm, HighTonePower_dBm) - HighTonePower_dBm, The degree of the high frequency crossmodulation product with respect to the strongest main signal.

        public float IP3_dBm;               // IP3_dBm

    };

    /*IP2 test structure*/
    [StructLayout(LayoutKind.Sequential)]
    public struct TraceAnalysisResult_IP2_TypeDef
    {
        public double LowToneFreq;          // Bass frequency, in units depending on the data source.
        public double HighToneFreq;         // High-pitched signal frequency, in units depending on the data source.
        public double IM2PFreq;             // Low frequency intermodulation frequency, unit dependent on data source.

        public float LowTonePower_dBm;      // Bass power, dBm.
        public float HighTonePower_dBm;     // Treble power, dBm.
        public float TonePowerDiff_dB;      // Bass power - treble power.
        public float IM2P_dBc;              // IM2P_dBc = max(LowTonePower_dBm, HighTonePower_dBm) - IM2P_dBm, the intensity of the low frequency crossmodulated product relative to the strongest main signal.

        public float IP2_dBm;               // IP2_dBm

    };

    /*AudioAnalysis structure*/
    [StructLayout(LayoutKind.Sequential)]
    public struct DSP_AudioAnalysis_TypeDef
    {
        public double AudioVoltage;   // Audio voltage (V).
        public double AudioFrequency; // Audio frequency (Hz).
        public double SINDA;          // Shinard (dB).
        public double THD;            // Total harmonic distortion (%).

    };

    /*XdB bandwidth information structure (Return)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct TraceAnalysisResult_XdB_TypeDef
    {
        public double XdBBandWidth_Hz; // XdB bandwidth (Hz).
        public double CenterFreq_Hz;   // Center frequency of XdB bandwidth (Hz).
        public double StartFreq_Hz;    // Starting frequency of XdB bandwidth (Hz).
        public double StopFreq_Hz;     // Termination frequency of XdB bandwidth (Hz).
        public float StartPower_dBm;   // Power for starting frequency of XdB bandwidth (dBm).
        public float StopPower_dBm;    // Power for terminating frequency of XdB bandwidth (dBm).
        public uint32_t PeakIndex;     // Peak index within the XdB bandwidth.
        public double PeakFreq_Hz;     // Peak frequency in XdB bandwidth (Hz).
        public float PeakPower_dBm;    // Peak power in XdB bandwidth (dBm).

    };

    /*Occupied bandwidth information structure (return)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct TraceAnalysisResult_OBW_TypeDef
    {
        public double OccupiedBandWidth; // Bandwidth of Occupied Bandwidth (Hz).
        public double CenterFreq_Hz;     // Occupied bandwidth center frequency (Hz).
        public double StartFreq_Hz;      // Occupied bandwidth start frequency (Hz).
        public double StopFreq_Hz;       // Occupied bandwidth stop frequency (Hz).
        public float StartPower_dBm;     // The power corresponding to the start frequency of the occupied bandwidth (dBm).
        public float StopPower_dBm;      // The power corresponding to the stop frequency of the occupied bandwidth (dBm).
        public float StartRatio;         // The power ratio corresponding the to start frequency of the occupied bandwidth (dBm).
        public float StopRatio;          // The power ratio corresponding the to stop frequency of the occupied bandwidth (dBm).
        public uint32_t PeakIndex;       // Peak index in the occupied bandwidth.
        public double PeakFreq_Hz;       // Peak frequency in the occupied bandwidth (Hz).
        public float PeakPower_dBm;      // Peak power in the occupied bandwidth (dBm).

    };

    /*Adjacent channel power ratio frequency information structure (input)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct DSP_ACPRFreqInfo_TypeDef
    {
        public double RBW;                 // Resolution bandwidth (Hz).
        public double MainChCenterFreq_Hz; // Main channel center frequency (Hz).
        public double MainChBW_Hz;         // Main channel bandwidth (Hz).
        public double AdjChSpace_Hz;       // Adjacent channel interval, difference between center frequency of main channel and center frequency of adjacent channel (Hz).
        public uint32_t AdjChPair;         // Adjacent channel pairs.1 represents the left and right 2 adjacent channels, and 2 represents the left and right 4 adjacent channels

    };

    /*Adjacent channel power ratio information structure (return)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct TraceAnalysisResult_ACPR_TypeDef
    {
        public float MainChPower_dBm;       // Main channel power (dBm).
        public uint32_t MainChPeakIndex;    // Peak index of the main channel.
        public double MainChPeakFreq_Hz;    // Peak frequency of the main channel (Hz).
        public float MainChPeakPower_dBm;   // Main channel peak power (dBm).

        public double L_AdjChCenterFreq_Hz; // Left adjacent channel center frequency (Hz).
        public double L_AdjChBW_Hz;         // Left adjacent channel bandwidth (Hz).
        public float L_AdjChPower_dBm;      // Left adjacent channel power (dBm).
        public float L_AdjChPowerRatio;     // Left adjacent channel power ratio (left adjacent channel power/main channel power).
        public float L_AdjChPowerDiff_dBc;  // Left Neighbor power difference (Left Neighbor Power - Main Channel power dBc).
        public float L_AdjChPeakIndex;      // Peak index of the left adjacent track.
        public double L_AdjChPeakFreq_Hz;   // Peak frequency of the left channel (Hz).
        public float L_AdjChPeakPower_dBm;  // Value power of left adjacent channel (dBm).

        public double R_AdjChCenterFreq_Hz; // Right adjacent channel center frequency (Hz).
        public double R_AdjChBW_Hz;         // Right adjacent channel bandwidth (Hz).
        public float R_AdjChPower_dBm;      // Right adjacent channel power (dBm).
        public float R_AdjChPowerRatio;     // Right neighbor power ratio (right neighbor power/main channel power).
        public float R_AdjChPowerDiff_dBc;  // Right Neighbor power difference (Right Neighbor Power - Main Channel power dBc).
        public float R_AdjChPeakIndex;      // Peak index of the right adjacent track.
        public double R_AdjChPeakFreq_Hz;   // Right channel peak frequency (Hz).
        public float R_AdjChPeakPower_dBm;  // Value power of right adjacent channel (dBm).
    };

    /*Channel power structure (return)*/
    [StructLayout(LayoutKind.Sequential)]
    public struct DSP_ChannelPowerInfo_TypeDef
    {
        public float ChannelPower_dBm;     // Channel power (dBm).
        public float PowerDensity;         // Channel power density (dBm/Hz).
        public float ChannelPeakIndex;     // Peak index within the channel.
        public double ChannelPeakFreq_Hz;  // Peak frequency in channel (Hz).
        public float ChannelPeakPower_dBm; // Peak power in channel (dBm).

    };

    /*Trace smoothing structure*/
    public enum SmoothMethod_TypeDef
    {
        MovingAvrage = 0x00, // Moving average
        MedianFilter = 0x01, // Median filtering
        PolynomialFit = 0x02 // Polynomial fit
    };

    public enum SWPApplication_TypeDef
    {
        SWPPhaseNoiseMeas = 0x00,   //phase noise measurement.
        SWPNoiseMeas = 0x01,        //noise measurement.
        SWPChannelPowerMeas = 0x02, //chanel power measurement.
        SWPOBWMeas = 0x03,          //occupation bandwidth measurement.
        SWPACPRMeas = 0x04,         //adjacent channel power rate measurement.
        SWPIM3Meas = 0x05           //IP3/IM3 measurement.
    };

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct DSP_SEMSegmentProfile_TypeDef
    {
        [MarshalAs(UnmanagedType.I1)]
        public bool mState;			// Off = 0; enable = 1;

        public double mStartFreq;	// Start frequency
        public double mLimitStart;	// Start threshold
        public double mStopFreq;	// Cut-off frequency
        public double mLimitStop;	// Cut-off threshold

        public int mMode;			// Absolute = 0; relative = 1
        public int mPriority;		// Requirement = 0; Recommendation = 1
    };

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct DSP_SEMProfile_TypeDef
    {
        public int mRefSetType;			// Reference setting mode: 0 Peak 1 Manual
        public float mManualRefLevel;	// Manually set the reference

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 16)]
        public DSP_SEMSegmentProfile_TypeDef[] mSegments;		// Configuration of each segment
    };

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct DSP_SEMSegmentResult_TypeDef
    {
        public double mLowerFreq;		// Low-end frequency
        public float mLowerLevel;		// Low end level
        public float mLowerMargin;		// Low end margin
        [MarshalAs(UnmanagedType.I1)]
        public bool mLowerPassOrFail;	// Low end pass, fail. 0: pass; 1: fail

        public double mUpperFreq;		// High-end frequency
        public float mUpperLevel;		// High end level
        public float mUpperMargin;		// High-end margin
        [MarshalAs(UnmanagedType.I1)]
        public bool mUpperPassOrFail;	// High end pass, fail
    };

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct DSP_SEMResult_TypeDef
    {
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 16)]
        public DSP_SEMSegmentResult_TypeDef[] mSegmentResults;	// Measurement results
        public DSP_SEMProfile_TypeDef mProfile;					// Measurement configuration
        public float mRefLevel;									// Reference used for measurement
    };

    [StructLayout(LayoutKind.Sequential)]
    public struct AMDemodParam_TypeDef
    {
        public IntPtr DemodWaveform;        // float* Demodulated waveform (%)
        public IntPtr AFSpectrum_ModDepth;  // float* Audio spectrum magnitude (%)
        public IntPtr AFSpectrum_Freq;      // double* Audio spectrum frequency (Hz)
        public uint DemodWaveformSize;      // Number of points in the demodulated waveform

        public float ModDepth;              // Modulation depth (%)
        public float ModDepthPeakPos;       // Positive peak modulation depth, Peak+ (%)
        public float ModDepthPeakNeg;       // Negative peak modulation depth average, Peak- (%)
        public float ModDepthHalfPeak;      // Half-peak modulation depth average, (Peak+ - Peak-)/2 (%)
        public float ModDepthRMS;           // RMS modulation depth (%)

        public float CarrierPower;          // Carrier power (dBm)
        public double ModRate;              // Modulation rate (Hz)
        public float SINAD;                 // SINAD (dB)
        public float RMSPower;              // RMS power (dBm)
        public double FreqError;            // Frequency error (Hz)
        public float SNR;                   // Signal-to-noise ratio (dB)
        public float DistTotalVrms;         // Distortion voltage ratio (RMS) (%)
        public float THD;                   // Total harmonic distortion (%)
        public float PEP;                   // Peak envelope power (dBm)
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct FMDemodParam_TypeDef
    {
        public IntPtr DemodWaveform;         // float* Demodulated waveform (Hz)
        public IntPtr AFSpectrum_Deviation;  // float* Audio spectrum magnitude (Hz)
        public IntPtr AFSpectrum_Freq;       // double* Audio spectrum frequency (Hz)
        public uint DemodWaveformSize;       // Number of points in the demodulated waveform

        public float Deviation;              // Modulation deviation (Hz)
        public float DeviationPeakPos;       // Positive peak deviation, Peak+ (Hz)
        public float DeviationPeakNeg;       // Negative peak deviation average, Peak- (Hz)
        public float DeviationHalfPeak;      // Half-peak deviation average, (Peak+ - Peak-)/2 (Hz)
        public float DeviationRMS;           // RMS deviation (Hz)

        public float CarrierPower;           // Carrier power (dBm)
        public double CarrierFreqErr;        // Carrier frequency error (Hz)
        public double ModRate;               // Modulation rate (Hz)

        public float SINAD;                  // SINAD (dB)
        public float SNR;                    // Signal-to-noise ratio (dB)
        public float DistTotalVrms;          // Distortion ratio (RMS) (%)
        public float THD;                    // Total harmonic distortion (%)
    }


    /* -------------------------------------------Device control related interface------------------------------------------------------ */
    [DllImport("htra_api.dll", EntryPoint = "Get_APIVersion", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Get_APIVersion();

    /* The dev interface to open the device, which must be called to obtain device resources before operating on the device  */
    [DllImport("htra_api.dll", EntryPoint = "Device_Open", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_Open(ref IntPtr Device, int DeviceNum, ref BootProfile_TypeDef BootProfile, ref BootInfo_TypeDef BootInfo);

    /* The dev interface to shut down the device. This function must be called to release device resources after the operation on the device is complete */
    [DllImport("htra_api.dll", EntryPoint = "Device_Close", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_Close(ref IntPtr Device);

    /* Device interface: Forces the data interface (such as USB) to re-enumerate. After this function succeeds, you need to call Device_Close() first, and then call Device_Open() to restore the connection with the device. */
    [DllImport("htra_api.dll", EntryPoint = "Device_ReEnumerate", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_ReEnumerate(ref IntPtr Device);

    /* The dev interface to set the system power supply status*/
    [DllImport("htra_api.dll", EntryPoint = "Device_SetSysPowerState", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_SetSysPowerState(ref IntPtr Device, SysPowerState_TypeDef SysPowerState);

    /* The dev interface to obtain device information, including device serial number, software and hardware version and other related information, non-real-time mode, without interrupting data acquisition, but information is only updated after obtaining data packets */
    [DllImport("htra_api.dll", EntryPoint = "Device_QueryDeviceInfo", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_QueryDeviceInfo(ref IntPtr Device, ref DeviceInfo_TypeDef DeviceInfo);

    /* The dev interface to obtain device information, including device serial number, software and hardware version and other related information, real-time mode, and a short period of time will occupy the data channel */
    [DllImport("htra_api.dll", EntryPoint = "Device_QueryDeviceInfo_Realtime", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_QueryDeviceInfo_Realtime(ref IntPtr Device, ref DeviceInfo_TypeDef DeviceInfo);

    /* The dev interface to obtain device status, including device temperature, hardware working status, geo-time information (optional support is required), etc., non-real-time mode, without interrupting data acquisition, but information is only updated after obtaining data packets */
    [DllImport("htra_api.dll", EntryPoint = "Device_QueryDeviceState", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_QueryDeviceState(ref IntPtr Device, ref DeviceState_TypeDef DeviceState);

    /* The dev interface to obtain device status, including device temperature, hardware working status, geo-time information (optional support is required), etc., real-time mode, and a short period of time will occupy the data channel */
    [DllImport("htra_api.dll", EntryPoint = "Device_QueryDeviceState_Realtime", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_QueryDeviceState_Realtime(ref IntPtr Device, ref DeviceState_TypeDef DeviceState);

    /* The dev interface, calibrates the reference clock frequency by external trigger signal of known clock period (advanced applications) ExtTriggerPeriod_s: indicates trigger signal period, ExtTriggerCount: indicates trigger number, RewriteRFCal: Indicates whether the calibrated reference clock frequency is rewritten to the calibration file as the default value for subsequent startup (irreversible). RefCLKFreq_Hz: indicates the calibrated reference clock frequency*/
    [DllImport("htra_api.dll", EntryPoint = "Device_CalibrateRefClock", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_CalibrateRefClock(ref IntPtr Device, ClkCalibrationSource_TypeDef ClkCalibrationSource, double TriggerPeriod_s, uint64_t TriggerCount, bool RewriteRFCal, ref double RefCLKFreq_Hz);



    /* The dev interface to update device firmware (advanced application)*/
    [DllImport("htra_api.dll", EntryPoint = "Device_UpdateFirmware", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_UpdateFirmware(int DeviceNum, Firmware_TypeDef Firmware, byte[] path);

    /* The dev interface to control the fan mode of the device*/
    [DllImport("htra_api.dll", EntryPoint = "Device_SetFanState", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_SetFanState(ref IntPtr Device, FanState_TypeDef FanState, float ThreshouldTemperature);

    /* The dev interface, set frequency response compensation*/
    [DllImport("htra_api.dll", EntryPoint = "Devcie_SetFreqResponseCompensation", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Devcie_SetFreqResponseCompensation(ref IntPtr Device, uint8_t State, ref double Frequency_Hz, ref float CorrectVal_dB, uint8_t Points);

    /*The dev interface,to set GNSS antenna status*/
    [DllImport("htra_api.dll", EntryPoint = "Device_SetGNSSAntennaState", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_SetGNSSAntennaState(ref IntPtr Device, GNSSAntennaState_TypeDef GNSSAntennaState);

    /* dev interface, obtains GNSS antenna status (opt.), non-real-time mode, continuous data acquisition, but the information is only updated after obtaining the data packet*/
    [DllImport("htra_api.dll", EntryPoint = "Device_GetGNSSAntennaState", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetGNSSAntennaState(ref IntPtr Device, ref GNSSAntennaState_TypeDef GNSSAntennaState);

    /* dev interface, obtains GNSS antenna status (opt.), real-time data acquisition, data channel will be occupied by a short period of time*/
    [DllImport("htra_api.dll", EntryPoint = "Device_GetGNSSAntennaState_Realtime", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetGNSSAntennaState_Realtime(ref IntPtr Device, ref GNSSAntennaState_TypeDef GNSSAntennaState);

    /* dev interface, parsing GNSS time and date information (option support required)*/
    [DllImport("htra_api.dll", EntryPoint = "Device_AnysisGNSSTime", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_AnysisGNSSTime(double ABSTimestamp, ref int16_t hour, ref int16_t minute, ref int16_t second, ref int16_t Year, ref int16_t month, ref int16_t day);

    /* dev interface, obtains GNSS altitude (opt.), non-real-time mode, continuous data acquisition, but the information is only updated after obtaining the data packet*/
    [DllImport("htra_api.dll", EntryPoint = "Device_GetGNSSAltitude", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetGNSSAltitude(ref IntPtr Device, ref int16_t Altitude);

    /* dev interface, obtains GNSS altitude (opt.), real-time data acquisition, data channel will be occupied by a short period of time*/
    [DllImport("htra_api.dll", EntryPoint = "Device_GetGNSSAltitude_Realtime", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetGNSSAltitude_Realtime(ref IntPtr Device, ref int16_t Altitude);

    /*dev interface, set the DOCXO working status in GNSS */
    [DllImport("htra_api.dll", EntryPoint = "Device_SetDOCXOWorkMode", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_SetDOCXOWorkMode(ref IntPtr Device, DOCXOWorkMode_TypeDef DOCXOWorkMode);

    /* dev interface, obtains GNSS DOCXO working state (opt.), real-time data acquisition, data channel will be occupied by a short period of time*/
    [DllImport("htra_api.dll", EntryPoint = "Device_GetDOCXOWorkMode_Realtime", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetDOCXOWorkMode_Realtime(ref IntPtr Device, ref DOCXOWorkMode_TypeDef DOCXOWorkMode);

    /* dev interface, obtains GNSS DOCXO working state (opt.), non-real-time mode, continuous data acquisition, but the information is only updated after obtaining the data packet*/
    [DllImport("htra_api.dll", EntryPoint = "Device_GetDOCXOWorkMode", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetDOCXOWorkMode(ref IntPtr Device, ref DOCXOWorkMode_TypeDef DOCXOWorkMode);

    /* dev interface, obtains GNSS DOCXO device state (opt.), non-real-time mode, continuous data acquisition, but the information is only updated after obtaining the data packet*/
    [DllImport("htra_api.dll", EntryPoint = "Device_GetGNSSInfo", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetGNSSInfo(ref IntPtr Device, ref GNSSInfo_TypeDef GNSSInfo);

    /*dev interface, obtains GNSS device state (opt.), real-time data acquisition, data channel will be occupied by a short period of time*/
    [DllImport("htra_api.dll", EntryPoint = "Device_GetGNSSInfo_Realtime", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetGNSSInfo_Realtime(ref IntPtr Device, ref GNSSInfo_TypeDef GNSSInfo);

    /* dev interface, obtain the IP addresses and subnet masks of all network devices on the network*/
    [DllImport("htra_api.dll", EntryPoint = "Device_GetNetworkDeviceList", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetNetworkDeviceList(ref uint8_t DeviceCount, NetworkDeviceInfo_TypeDef[] DeviceInfo, uint8_t[] LocalIP, uint8_t[] LocalMask);

    /* dev interface, configure the IP address and subnet mask of the network device based on the device UID*/
    [DllImport("htra_api.dll", EntryPoint = "Device_SetNetworkDeviceIP", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_SetNetworkDeviceIP(uint64_t DeviceUID, uint8_t[] IPAddress, uint8_t[] SubnetMask);

    /* dev interface, this section describes how to configure the IP address and subnet mask of the network device based on the device IP address*/
    [DllImport("htra_api.dll", EntryPoint = "Device_SetNetworkDeviceIP_PM1", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_SetNetworkDeviceIP_PM1(uint8_t[] DeviceIP, uint8_t[] IPAddress, uint8_t[] SubnetMask);

    /* dev interface, get the full UID information, in a non-real-time manner, without interrupting data acquisition, but the information is only updated after the data packet is obtained*/
    [DllImport("htra_api.dll", EntryPoint = "Device_GetFullUID", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetFullUID(ref IntPtr Device, ref uint64_t UID_L64, ref uint32_t UID_H32);

    /* dev interface, get GNSS signal-to-noise ratio (optional support required), in a non-real-time manner, without interrupting data acquisition, but the information is only updated after the data packet is obtained*/
    [DllImport("htra_api.dll", EntryPoint = "Device_GetGNSS_SatDate", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetGNSS_SatDate(ref IntPtr Device, ref GNSS_SatDate_TypeDef GNSS_SatDate);

    /* dev interface, obtain GNSS signal to noise ratio (optional support required) in real time, which occupies the data channel for a short time*/
    [DllImport("htra_api.dll", EntryPoint = "Device_GetGNSS_SatDate_Realtime", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetGNSS_SatDate_Realtime(ref IntPtr Device, ref GNSS_SatDate_TypeDef GNSS_SatDate);

    /* dev interface, update GNSS Firmware (Advanced Applications)*/
    [DllImport("htra_api.dll", EntryPoint = "Device_GNSSFirmwareUpdate", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GNSSFirmwareUpdate(int DeviceNum, ref char path);

    // Device_GetHardwareState Mapping
    [DllImport("htra_api.dll", EntryPoint = "Device_GetHardwareState", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetHardwareState(ref IntPtr Device, ref HardWareState_TypeDef HardWareState);

    // Device_QueryDeviceInfoWithBus Mapping
    [DllImport("htra_api.dll", EntryPoint = "Device_QueryDeviceInfoWithBus", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_QueryDeviceInfoWithBus(int DeviceNum, ref BootProfile_TypeDef BootProfile, ref BootInfo_TypeDef BootInfo);

    // Device_SetFreqScan Mapping
    [DllImport("htra_api.dll", EntryPoint = "Device_SetFreqScan", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_SetFreqScan(ref IntPtr Device, double StartFreq_Hz, double StopFreq_Hz, ushort SweepPts);
    /* -------------------------------------------The following is the SWP mode (SWEEP related interface)------------------------------------------------------ */
    /* SWP mode, configure SWP_Profile to the default value */
    [DllImport("htra_api.dll", EntryPoint = "SWP_ProfileDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SWP_ProfileDeInit(ref IntPtr Device, ref SWP_Profile_TypeDef UserProfile_O);
    [DllImport("htra_api.dll", EntryPoint = "SWP_EZProfileDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SWP_EZProfileDeInit(ref IntPtr Device, ref SWP_EZProfile_TypeDef UserProfile_O);

    /* SWP mode, set SWP mode parameters */
    [DllImport("htra_api.dll", EntryPoint = "SWP_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SWP_Configuration(ref IntPtr Device, ref SWP_Profile_TypeDef ProfileIn, ref SWP_Profile_TypeDef ProfileOut, ref SWP_TraceInfo_TypeDef TraceInfo);
    [DllImport("htra_api.dll", EntryPoint = "SWP_EZConfiguration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SWP_EZConfiguration(ref IntPtr Device, ref SWP_EZProfile_TypeDef ProfileIn, ref SWP_EZProfile_TypeDef ProfileOut, ref SWP_TraceInfo_TypeDef TraceInfo);

    /* SWP mode, obtain partial scanning spectrum data in SWP mode */
    [DllImport("htra_api.dll", EntryPoint = "SWP_GetPartialSweep", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SWP_GetPartialSweep(ref IntPtr Device, double[] Freq_Hz, float[] PowerSpec_dBm, ref int HopIndex, ref int FrameIndex, ref MeasAuxInfo_TypeDef MeasAuxInfo);

    /* SWP mode, obtain the completely refreshed spectrum data in SWP mode */
    [DllImport("htra_api.dll", EntryPoint = "SWP_GetFullSweep", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SWP_GetFullSweep(ref IntPtr Device, double[] Freq_Hz, float[] PowerSpec_dBm, ref MeasAuxInfo_TypeDef MeasAuxInfo);

    /*SWP mode: Obtain spectrum data that is partially refreshed but contains non-refreshed data in SWP mode (advanced application) */
    [DllImport("htra_api.dll", EntryPoint = "SWP_GetPartialUpdatedFullSweep", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SWP_GetPartialUpdatedFullSweep(ref IntPtr Device, double[] Freq_Hz, float[] PowerSpec_dBm, ref int HopIndex, ref int FrameIndex, ref MeasAuxInfo_TypeDef MeasAuxInfo);

    /* SWP mode: Reset MaxHold Spectrum data in SWP mode (advanced application) */
    [DllImport("htra_api.dll", EntryPoint = "SWP_GetAmplitudeCalibrationTrace", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SWP_GetAmplitudeCalibrationTrace(ref IntPtr Device, double[] Frequency_Hz, float[] Compensate_dB, ref float CalTemperature_C);

    /*  SWP mode, reset spectrum data of MaxHold and MinHold in SWP mode(advanced application) */
    [DllImport("htra_api.dll", EntryPoint = "SWP_ResetTraceHold", CallingConvention = CallingConvention.Cdecl)]
    public static extern void SWP_ResetTraceHold(ref IntPtr Device);

    /* SWP mode: Give the recommended device configuration based on the application goal */
    [DllImport("htra_api.dll", EntryPoint = "SWP_AutoSet", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SWP_AutoSet(ref IntPtr Device, SWPApplication_TypeDef Application, ref SWP_Profile_TypeDef ProfileIn, ref SWP_Profile_TypeDef ProfileOut, ref SWP_TraceInfo_TypeDef TraceInfo, uint8_t ifDoConfig);

    /* -------------------------------------------The following is the time domain stream mode(IQS related interface)------------------------------------------------------ */

    /*IQS mode, set IQS_Profile to the default value */
    [DllImport("htra_api.dll", EntryPoint = "IQS_ProfileDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_ProfileDeInit(ref IntPtr Device, ref IQS_Profile_TypeDef UserProfile_O);
    [DllImport("htra_api.dll", EntryPoint = "IQS_EZProfileDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_EZProfileDeInit(ref IntPtr Device, ref IQS_EZProfile_TypeDef UserProfile_O);

    /* IQS mode, set IQS mode parameters */
    [DllImport("htra_api.dll", EntryPoint = "IQS_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_Configuration(ref IntPtr Device, ref IQS_Profile_TypeDef ProfileIn, ref IQS_Profile_TypeDef ProfileOut, ref IQS_StreamInfo_TypeDef StreamInfo);
    [DllImport("htra_api.dll", EntryPoint = "IQS_EZConfiguration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_EZConfiguration(ref IntPtr Device, ref IQS_EZProfile_TypeDef ProfileIn, ref IQS_EZProfile_TypeDef ProfileOut, ref IQS_StreamInfo_TypeDef StreamInfo);

    /* IQS mode, initiating the bus trigger */
    [DllImport("htra_api.dll", EntryPoint = "IQS_BusTriggerStart", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_BusTriggerStart(ref IntPtr Device);

    /* IQS mode, terminating the bus trigger */
    [DllImport("htra_api.dll", EntryPoint = "IQS_BusTriggerStop", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_BusTriggerStop(ref IntPtr Device);

    /*IQS mode, wait for multi-machine synchronization trigger signal*/
    [DllImport("htra_api.dll", EntryPoint = "IQS_MultiDevice_WaitExternalSync", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_MultiDevice_WaitExternalSync(ref IntPtr Device, ref IQS_Profile_TypeDef ProfileIn);

    /* IQS mode, multiple devices can run synchronously*/
    [DllImport("htra_api.dll", EntryPoint = "IQS_MultiDevice_Run", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_MultiDevice_Run(ref IntPtr Device);

    /* IQS mode, the interwoven IQ data stream under IQS mode is obtained, and the scale factor from integer to absolute amplitude (V units) and the related information of trigger are obtained */
    [DllImport("htra_api.dll", EntryPoint = "IQS_GetIQStream", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_GetIQStream(ref IntPtr Device, ref IntPtr AlternIQStream, ref float ScaleToV, ref IQS_TriggerInfo_TypeDef TriggerInfo, ref MeasAuxInfo_TypeDef MeasAuxInfo);
    [DllImport("htra_api.dll", EntryPoint = "IQS_GetIQStream_PM1", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_GetIQStream_PM1(ref IntPtr Device, ref IQStream_TypeDef IQStream);
    [DllImport("htra_api.dll", EntryPoint = "IQS_GetIQStream_PM2", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_GetIQStream_PM2(ref IntPtr Device, ref IQStream_TypeDef IQStream, ref MeasAuxInfo_TypeDef MeasAuxInfo);
    [DllImport("htra_api.dll", EntryPoint = "IQS_GetIQStream_Data", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_GetIQStream_Data(ref IntPtr Device, int16_t[] IQ_data);

    /*IQS mode，initiates timer - External triggers single synchronization*/
    [DllImport("htra_api.dll", EntryPoint = "IQS_SyncTimer", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_SyncTimer(ref IntPtr Device);

    /* -------------------------------------------The following is the detection mode (DET correlation interface)------------------------------------------------------ */

    /* DET mode, set DET_Profile to the default value */
    [DllImport("htra_api.dll", EntryPoint = "DET_ProfileDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DET_ProfileDeInit(ref IntPtr Device, ref DET_Profile_TypeDef UserProfile_O);
    [DllImport("htra_api.dll", EntryPoint = "DET_EZProfileDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DET_EZProfileDeInit(ref IntPtr Device, ref DET_EZProfile_TypeDef UserProfile_O);

    /* DET mode, set parameters related to the DET mode */
    [DllImport("htra_api.dll", EntryPoint = "DET_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DET_Configuration(ref IntPtr Device, ref DET_Profile_TypeDef ProfileIn, ref DET_Profile_TypeDef ProfileOut, ref DET_StreamInfo_TypeDef StreamInfo);
    [DllImport("htra_api.dll", EntryPoint = "DET_EZConfiguration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DET_EZConfiguration(ref IntPtr Device, ref DET_EZProfile_TypeDef ProfileIn, ref DET_EZProfile_TypeDef ProfileOut, ref DET_StreamInfo_TypeDef StreamInfo);

    /* ZeroSpan mode, configure ZeroSpan mode related parameters */
    [DllImport("htra_api.dll", EntryPoint = "ZSP_ProfileDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ZSP_ProfileDeInit(ref IntPtr Device, ref ZSP_Profile_TypeDef UserProfile_O);
    [DllImport("htra_api.dll", EntryPoint = "ZSP_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ZSP_Configuration(ref IntPtr Device, ref ZSP_Profile_TypeDef ProfileIn, ref ZSP_Profile_TypeDef ProfileOut, ref DET_StreamInfo_TypeDef StreamInfo);

    /* DET mode, initiating the bus trigger */
    [DllImport("htra_api.dll", EntryPoint = "DET_BusTriggerStart", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DET_BusTriggerStart(ref IntPtr Device);

    /* DET mode, terminating the bus trigger */
    [DllImport("htra_api.dll", EntryPoint = "DET_BusTriggerStop", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DET_BusTriggerStop(ref IntPtr Device);

    /* DET mode, obtain the detection data in DET mode, and obtain the scale factor from integer to absolute amplitude (V units) and trigger related information. NormalizedPowerStream is the square root of I square +Q square */
    [DllImport("htra_api.dll", EntryPoint = "DET_GetPowerStream", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DET_GetPowerStream(ref IntPtr Device, float[] NormalizedPowerStream, ref float ScaleToV, ref DET_TriggerInfo_TypeDef TriggerInfo, ref MeasAuxInfo_TypeDef MeasAuxInfo);

    /* DET mode, Initiates the timer - External triggers a single synchronization*/
    [DllImport("htra_api.dll", EntryPoint = "DET_SyncTimer", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DET_SyncTimer(ref IntPtr Device);

    /* -------------------------------------------The following is the real-time spectrum analysis mode (RTA related interface)------------------------------------------------------ */

    /* RTA mode, set RTA_Profile to the default value */
    [DllImport("htra_api.dll", EntryPoint = "RTA_ProfileDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int RTA_ProfileDeInit(ref IntPtr Device, ref RTA_Profile_TypeDef UserProfile_O);
    [DllImport("htra_api.dll", EntryPoint = "RTA_EZProfileDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int RTA_EZProfileDeInit(ref IntPtr Device, ref RTA_EZProfile_TypeDef UserProfile_O);

    /* RTA mode, set parameters related to the RTA mode */
    [DllImport("htra_api.dll", EntryPoint = "RTA_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int RTA_Configuration(ref IntPtr Device, ref RTA_Profile_TypeDef ProfileIn, ref RTA_Profile_TypeDef ProfileOut, ref RTA_FrameInfo_TypeDef FrameInfo);
    [DllImport("htra_api.dll", EntryPoint = "RTA_EZConfiguration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int RTA_EZConfiguration(ref IntPtr Device, ref RTA_EZProfile_TypeDef ProfileIn, ref RTA_EZProfile_TypeDef ProfileOut, ref RTA_FrameInfo_TypeDef FrameInfo);

    /* RTA mode, initiating the bus trigger */
    [DllImport("htra_api.dll", EntryPoint = "RTA_BusTriggerStart", CallingConvention = CallingConvention.Cdecl)]
    public static extern int RTA_BusTriggerStart(ref IntPtr Device);

    /* RTA mode, terminating the bus trigger */
    [DllImport("htra_api.dll", EntryPoint = "RTA_BusTriggerStop", CallingConvention = CallingConvention.Cdecl)]
    public static extern int RTA_BusTriggerStop(ref IntPtr Device);

    /* RTA mode, real-time spectrum data and trigger related information are obtained in RTA mode */
    [DllImport("htra_api.dll", EntryPoint = "RTA_GetRealTimeSpectrum", CallingConvention = CallingConvention.Cdecl)]
    public static extern int RTA_GetRealTimeSpectrum(ref IntPtr Device, uint8_t[] SpectrumStream, uint16_t[] SpectrumBitmap, ref RTA_PlotInfo_TypeDef PlotInfo, ref RTA_TriggerInfo_TypeDef TriggerInfo, ref MeasAuxInfo_TypeDef MeasAuxInfo);

    /*RTA mode, to obtain real-time spectrum (without probability density map) and trigger related information in RTA mode*/
    [DllImport("htra_api.dll", EntryPoint = "RTA_GetRealTimeSpectrum_Raw", CallingConvention = CallingConvention.Cdecl)]
    public static extern int RTA_GetRealTimeSpectrum_Raw(ref IntPtr Device, uint8_t[] SpectrumStream, ref RTA_PlotInfo_TypeDef PlotInfo, ref RTA_TriggerInfo_TypeDef TriggerInfo, ref MeasAuxInfo_TypeDef MeasAuxInfo);

    /*RTA mode, the trigger threshold is issued when the spectrum template is used to trigger. Input the number of concerns, the Freq array and the Power array correspond to each other, just input concerns, and the power between points will be automatically completed*/
    [DllImport("htra_api.dll", EntryPoint = "RTA_SpectrumMaskTrigger_SetMask", CallingConvention = CallingConvention.Cdecl)]
    public static extern int RTA_SpectrumMaskTrigger_SetMask(ref IntPtr Device, uint32_t NodeNum, double[] FreqHz_In, float[] PowerdBm_In, double[] FreqHz_Out, float[] PowerdBm_Out);

    /*RTA mode, initiating spectrum template trigger*/
    [DllImport("htra_api.dll", EntryPoint = "RTA_SpectrumMaskTrigger_Run", CallingConvention = CallingConvention.Cdecl)]
    public static extern int RTA_SpectrumMaskTrigger_Run(ref IntPtr Device);

    /*RTA mode, spectrum template trigger is stopped*/
    [DllImport("htra_api.dll", EntryPoint = "RTA_SpectrumMaskTrigger_Stop", CallingConvention = CallingConvention.Cdecl)]
    public static extern int RTA_SpectrumMaskTrigger_Stop(ref IntPtr Device);

    /*RTA mode, initiate timer - External trigger single synchronization*/
    [DllImport("htra_api.dll", EntryPoint = "RTA_SyncTimer", CallingConvention = CallingConvention.Cdecl)]
    public static extern int RTA_SyncTimer(ref IntPtr Device);

    /* -------------------------------------------The following are auxiliary signal generator controls (ASG related interfaces)------------------------------------------------------ */

    /* ASG to set ASG_Profile to the default value */
    [DllImport("htra_api.dll", EntryPoint = "ASG_ProfileDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ASG_ProfileDeInit(ref IntPtr Device, ref ASG_Profile_TypeDef Profile);

    /* ASG to set working status */
    [DllImport("htra_api.dll", EntryPoint = "ASG_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ASG_Configuration(ref IntPtr Device, ref ASG_Profile_TypeDef ProfileIn, ref ASG_Profile_TypeDef ProfileOut, ref ASG_Info_TypeDef ASG_Info);

    /* -------------------------------------------The following is Analog demodulation (Analog related interface)------------------------------------------------------ */

    /* Analog modulation, create */
    [DllImport("htra_api.dll", EntryPoint = "ASD_Open", CallingConvention = CallingConvention.Cdecl)]
    public static extern void ASD_Open(ref IntPtr AnalogMod);

    /* Analog modulation, close */
    [DllImport("htra_api.dll", EntryPoint = "ASD_Close", CallingConvention = CallingConvention.Cdecl)]
    public static extern void ASD_Close(ref IntPtr AnalogMod);

    /* Modulation and demodulation mode, FM demodulation */
    [DllImport("htra_api.dll", EntryPoint = "ASD_Demodulate_FM", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ASD_Demodulate_FM(ref IntPtr AnalogMod, ref IQStream_TypeDef IQStreamIn, bool reset, float[] result, ref double carrierOffsetHz);

    /*  Modulation and demodulation mode, AM demodulation */
    [DllImport("htra_api.dll", EntryPoint = "ASD_Demodulate_AM", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ASD_Demodulate_AM(ref IQStream_TypeDef IQStreamIn, float[] result);

    // Mapping FM demodulation functions
    [DllImport("htra_api.dll", EntryPoint = "ASD_FMDemodulation", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ASD_FMDemodulation(ref IntPtr AnalogMod, ref IQStream_TypeDef IQStreamIn, bool reset, float[] result, ref FM_DemodParam_TypeDef FM_DemodParam);

    // Mapping AM demodulation functions
    [DllImport("htra_api.dll", EntryPoint = "ASD_AMDemodulation", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ASD_AMDemodulation(ref IntPtr AnalogMod, ref IQStream_TypeDef IQStreamIn, bool reset, float[] result, ref AM_DemodParam_TypeDef AM_DemodParam);

    // Create an Analog Demod instance
    [DllImport("htra_api.dll", EntryPoint = "ADM_Open", CallingConvention = CallingConvention.Cdecl)]
    public static extern void ADM_Open(ref IntPtr ADM);

    // Release an Analog Demod instance
    [DllImport("htra_api.dll", EntryPoint = "ADM_Close", CallingConvention = CallingConvention.Cdecl)]
    public static extern void ADM_Close(ref IntPtr ADM);

    // AM demodulation, only returns the demodulated waveform
    [DllImport("htra_api.dll", EntryPoint = "ADM_AMDemod", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ADM_AMDemod(ref IntPtr ADM, IntPtr AlternIQStream, DataFormat_TypeDef DataFormat, ulong SamplePoints, double IQSampleRate, [Out] float[] DemodWaveform);

    // FM demodulation, only returns the demodulated waveform
    [DllImport("htra_api.dll", EntryPoint = "ADM_FMDemod", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ADM_FMDemod(ref IntPtr ADM, IntPtr AlternIQStream, DataFormat_TypeDef DataFormat, ulong SamplePoints, double IQSampleRate, [MarshalAs(UnmanagedType.I1)] bool reset, [Out] float[] DemodWaveform);

    // AM demodulation, returns demodulated waveform and modulation parameters
    [DllImport("htra_api.dll", EntryPoint = "ADM_AMDemod_PM1", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ADM_AMDemod_PM1(ref IntPtr ADM, IntPtr AlternIQStream, DataFormat_TypeDef DataFormat, ulong SamplePoints, double IQSampleRate, float IQS_ScaleToV, ref AMDemodParam_TypeDef AMDemodParam);

    // FM demodulation, returns demodulated waveform and modulation parameters
    [DllImport("htra_api.dll", EntryPoint = "ADM_FMDemod_PM1", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ADM_FMDemod_PM1(ref IntPtr ADM, IntPtr AlternIQStream, DataFormat_TypeDef DataFormat, ulong SamplePoints, double IQSampleRate, float IQS_ScaleToV, [MarshalAs(UnmanagedType.I1)] bool reset, ref FMDemodParam_TypeDef FMDemodParam);

    /* -------------------------------------------The following are digital signal processing auxiliary functions (DSP related auxiliary functions)------------------------------------------------------ */

    /* DSP auxiliary function to enable the DSP function */
    [DllImport("htra_api.dll", EntryPoint = "DSP_Open", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_Open(ref IntPtr DSP);

    /* DSP auxiliary function to initialize FFT parameters */
    [DllImport("htra_api.dll", EntryPoint = "DSP_FFT_DeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_FFT_DeInit(ref DSP_FFT_TypeDef IQToSpectrum);

    /* DSP auxiliary function to configurate FFT parameters */
    [DllImport("htra_api.dll", EntryPoint = "DSP_FFT_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_FFT_Configuration(ref IntPtr DSP, ref DSP_FFT_TypeDef ProfileIn, ref DSP_FFT_TypeDef ProfileOut, ref uint32_t TracePoints, ref double RBWRatio);

    /* DSP auxiliary function to convert IQ data into spectral data */
    [DllImport("htra_api.dll", EntryPoint = "DSP_FFT_IQSToSpectrum", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_FFT_IQSToSpectrum(ref IntPtr DSP, ref IQStream_TypeDef IQStream, double[] Freq_Hz, float[] PowerSpec_dBm);

    /* DSP auxiliary function to initialize the LPF parameter */
    [DllImport("htra_api.dll", EntryPoint = "DSP_LPF_DeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_LPF_DeInit(ref Filter_TypeDef LPF_ProfileIn);

    /*  DSP auxiliary function with low pass filter */
    [DllImport("htra_api.dll", EntryPoint = "DSP_LPF_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_LPF_Configuration(ref IntPtr DSP, ref Filter_TypeDef LPF_ProfileIn, ref Filter_TypeDef LPF_ProfileOut);

    /*  DSP auxiliary function to reset cache in LPF */
    [DllImport("htra_api.dll", EntryPoint = "DSP_LPF_Reset", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_LPF_Reset(ref IntPtr DSP);

    /* DSP auxiliary function, low pass filter _ real signal */
    [DllImport("htra_api.dll", EntryPoint = "DSP_LPF_Execute_Real", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_LPF_Execute_Real(ref IntPtr DSP, float[] Signal, float[] LPF_Signal);

    /* DSP auxiliary function, low pass filter _ complex signal */
    [DllImport("htra_api.dll", EntryPoint = "DSP_LPF_Execute_Complex", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_LPF_Execute_Complex(ref IntPtr DSP, ref IQStream_TypeDef IQStreamIn, ref IQStream_TypeDef IQStreamOut);

    /* DSP auxiliary function to initialize DDC configuration parameters */
    [DllImport("htra_api.dll", EntryPoint = "DSP_DDC_DeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_DDC_DeInit(ref DSP_DDC_TypeDef DDC_Profile);

    /* DSP auxiliary function to configure DDC  */
    [DllImport("htra_api.dll", EntryPoint = "DSP_DDC_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_DDC_Configuration(ref IntPtr DSP, ref DSP_DDC_TypeDef DDC_ProfileIn, ref DSP_DDC_TypeDef DDC_ProfileOut);

    /* DSP helper function to reset cache in DDC */
    [DllImport("htra_api.dll", EntryPoint = "DSP_DDC_Reset", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_DDC_Reset(ref IntPtr DSP);

    // Mapping the function to get the DDC delay
    [DllImport("htra_api.dll", EntryPoint = "DSP_DDC_GetDelay", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_DDC_GetDelay(ref IntPtr DSP, ref uint delay);

    /* DSP auxiliary function, DDC execution */
    [DllImport("htra_api.dll", EntryPoint = "DSP_DDC_Execute", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_DDC_Execute(ref IntPtr DSP, ref IQStream_TypeDef IQStreamIn, ref IQStream_TypeDef IQStreamOut);

    /*DSP auxiliary function to analyze the IM3 parameter of the trace */
    [DllImport("htra_api.dll", EntryPoint = "DSP_TraceAnalysis_IM3", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_TraceAnalysis_IM3(double[] Freq_Hz, float[] PowerSpec_dBm, uint32_t TracePoints, ref TraceAnalysisResult_IP3_TypeDef IM3Result);

    /*DSP auxiliary function to analyze the IM2 parameter of the trace */
    [DllImport("htra_api.dll", EntryPoint = "DSP_TraceAnalysis_IM2", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_TraceAnalysis_IM2(double[] Freq_Hz, float[] PowerSpec_dBm, uint32_t TracePoints, ref TraceAnalysisResult_IP2_TypeDef IM2Result);

    /* DSP auxiliary function to analyze the phase noise parameters of the trace */
    [DllImport("htra_api.dll", EntryPoint = "DSP_TraceAnalysis_PhaseNoise", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_TraceAnalysis_PhaseNoise(double[] Freq_Hz, float[] PowerSpec_dBm, double[] OffsetFreqs, uint32_t TracePoints, uint32_t OffsetFreqsToAnalysis, double[] CarrierFreqOut, float[] PhaseNoiseOut_dBc);

    /* DSP auxiliary function to analyze the channel power of the trace*/
    [DllImport("htra_api.dll", EntryPoint = "DSP_TraceAnalysis_ChannelPower", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_TraceAnalysis_ChannelPower(double[] Freq_Hz, float[] PowerSpec_dBm, uint32_t TracePoints, double CenterFrequency, double AnalysisSpan, double RBW, ref DSP_ChannelPowerInfo_TypeDef ChannelPowerResult);

    /* DSP auxiliary function to analyze the XdB bandwidth of the trace*/
    [DllImport("htra_api.dll", EntryPoint = "DSP_TraceAnalysis_XdBBW", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_TraceAnalysis_XdBBW(double[] Freq_Hz, float[] PowerSpec_dBm, uint32_t TracePoints, float XdB, ref TraceAnalysisResult_XdB_TypeDef XdBResult);

    /* DSP auxiliary function to analyze the occupied bandwidth of the trace*/
    [DllImport("htra_api.dll", EntryPoint = "DSP_TraceAnalysis_OBW", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_TraceAnalysis_OBW(double[] Freq_Hz, float[] PowerSpec_dBm, uint32_t TracePoints, float OccupiedRatio, ref TraceAnalysisResult_OBW_TypeDef OBWResult);

    /* DSP auxiliary function to analyze the adjacent track power ratio of the trace*/
    [DllImport("htra_api.dll", EntryPoint = "DSP_TraceAnalysis_ACPR", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_TraceAnalysis_ACPR(double[] Freq_Hz, float[] PowerSpec_dBm, uint32_t TracePoints, DSP_ACPRFreqInfo_TypeDef ACPRFreqInfo, ref TraceAnalysisResult_ACPR_TypeDef ACPRResult);

    /* Audio analysis function analyzes audio voltage (V), audio frequency (Hz), Sinard (dB) and total harmonic distortion (%) parameters  */
    [DllImport("htra_api.dll", EntryPoint = "DSP_AudioAnalysis", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_AudioAnalysis(double[] Audio, uint64_t SamplePoints, double SampleRate, ref DSP_AudioAnalysis_TypeDef AudioAnalysis);

    /*DSP auxiliary function, spectrum interception*/
    [DllImport("htra_api.dll", EntryPoint = "DSP_InterceptSpectrum", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_InterceptSpectrum(double StartFreq_Hz, double StopFreq_Hz, double[] Freq_Hz, float[] PowerSpec_dBm, uint32_t FullsweepTracePoints, double[] FrequencyOut, float[] PowerSpecOut_dBm, ref uint32_t InterceptPoints);

    /* DSP auxiliary function,  trace smooth */
    [DllImport("htra_api.dll", EntryPoint = "DSP_TraceSmooth", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_TraceSmooth([In, Out] float[] Trace, UInt32 TracePoints, SmoothMethod_TypeDef SmoothMethod, UInt32 IndexOffset, UInt32 WindowLength, UInt32 PolynomialOrder);

    /* DSP auxiliary function, disable the DSP function */
    [DllImport("htra_api.dll", EntryPoint = "DSP_Close", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_Close(ref IntPtr DSP);

    /* DSP auxiliary function, spectrum emission mask (SEM) analysis */
    [DllImport("htra_api.dll", EntryPoint = "DSP_SEMAnalysis", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_SEMAnalysis(ref DSP_SEMProfile_TypeDef semProfile, double[] Freq_Hz, float[] PowerSpec_dBm, uint TracePoints, double CenterFreq, ref DSP_SEMResult_TypeDef SEMResult);

    /* -------------------------------------------The following functions are deprecated, not yet available, and not necessary invoked------------------------------------------------------ */

    /* -------------------------------------------The following is the control of vector network analyzer (VNA related interface, not yet open)------------------------------------------------------ */

    /* VNA mode, set VNA_Profile to the default value */
    [DllImport("htra_api.dll", EntryPoint = "VNA_ProfileDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VNA_ProfileDeInit(ref IntPtr Device, ref VNA_Profile_TypeDef Profile);

    /* VNA mode, configurate the work state */
    [DllImport("htra_api.dll", EntryPoint = "VNA_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VNA_Configuration(ref IntPtr Device, ref VNA_Profile_TypeDef ProfileIn, ref VNA_Profile_TypeDef ProfileOut, ref VNA_Info_TypeDef VNA_Info);

    /* VNA mode, get the complete scan result */
    [DllImport("htra_api.dll", EntryPoint = "VNA_GetFullSweep", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VNA_GetFullSweep(ref IntPtr Device, double[] Frequency, double[] Result_dB, double[] Result_Deg);

    /* VNA mode, get the partial scan result */
    [DllImport("htra_api.dll", EntryPoint = "VNA_GetPartialSweep", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VNA_GetPartialSweep(ref IntPtr Device, ref double Frequency, ref double Result_dB, ref double Result_Deg, ref int Index);

    /* VNA mode, get single IQStream */
    [DllImport("htra_api.dll", EntryPoint = "VNA_GetIQStream", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VNA_GetIQStream(ref IntPtr Device, ref double I_Ref, ref double Q_Ref, ref double I_Meas, ref double Q_Meas, int DataLength, ref int PacketIndex);

    /* VNA mode, calibration */
    [DllImport("htra_api.dll", EntryPoint = "VNA_Calibration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VNA_Calibration(ref IntPtr Device, VNA_Calibration_TypeDef VNA_Calibration);

    /* VNA mode, calibration done */
    [DllImport("htra_api.dll", EntryPoint = "VNA_Calibration_Done", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VNA_Calibration_Done(ref IntPtr Device);

    /* VNA mode, save calibration data*/
    [DllImport("htra_api.dll", EntryPoint = "VNA_SaveCalibrationData", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VNA_SaveCalibrationData(ref IntPtr Device, byte[] Calibrationfile);
    [DllImport("htra_api.dll", EntryPoint = "VNA_LoadCalibrationData", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VNA_LoadCalibrationData(ref IntPtr Device, byte[] Calibratiofile);

    /* -------------------------------------------The following multi-profile mode (MPS related interface, not open yet)------------------------------------------------------ */

    /* MPS mode, set SWP mode parameters */
    [DllImport("htra_api.dll", EntryPoint = "MPS_SWP_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int MPS_SWP_Configuration(ref IntPtr Device, uint16_t ProfileNum, ref SWP_Profile_TypeDef ProfileIn, ref SWP_Profile_TypeDef ProfileOut, ref SWP_TraceInfo_TypeDef TraceInfo);

    /* MPS mode, set parameters related to RTA mode  */
    [DllImport("htra_api.dll", EntryPoint = "MPS_RTA_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int MPS_RTA_Configuration(ref IntPtr Device, uint16_t ProfileNum, ref RTA_Profile_TypeDef ProfileIn, ref RTA_Profile_TypeDef ProfileOut, ref RTA_FrameInfo_TypeDef FrameInfo);

    /* MPS mode, set IQS mode parameters */
    [DllImport("htra_api.dll", EntryPoint = "MPS_IQS_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int MPS_IQS_Configuration(ref IntPtr Device, uint16_t ProfileNum, ref IQS_Profile_TypeDef ProfileIn, ref IQS_Profile_TypeDef ProfileOut, ref IQS_StreamInfo_TypeDef StreamInfo);

    /* MPS mode: Set the parameters related to the DET mode */
    [DllImport("htra_api.dll", EntryPoint = "MPS_DET_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int MPS_DET_Configuration(ref IntPtr Device, uint16_t ProfileNum, ref DET_Profile_TypeDef ProfileIn, ref DET_Profile_TypeDef ProfileOut, ref DET_StreamInfo_TypeDef StreamInfo);

    /* MPS mode, start the MPS mode */
    [DllImport("htra_api.dll", EntryPoint = "MPS_Start", CallingConvention = CallingConvention.Cdecl)]
    public static extern int MPS_Start(ref IntPtr Device, uint16_t AllProfileNum);

    /* MPS mode, simple version to obtain data */
    [DllImport("htra_api.dll", EntryPoint = "MPS_GetData_Simple", CallingConvention = CallingConvention.Cdecl)]
    public static extern int MPS_GetData_Simple(ref IntPtr Device, ref MPSDataInfo_TypeDef MPSDataInfo, ref double Frequency, ref float PowerSpec_dBm, ref int16_t AlternIQStream, ref float DETTrace, ref uint8_t SpectrumTrace, ref uint16_t SpectrumBitmap);

    /* MPS mode, open up memory */
    [DllImport("htra_api.dll", EntryPoint = "MPS_DataBuffer_New", CallingConvention = CallingConvention.Cdecl)]
    public static extern int MPS_DataBuffer_New(ref IntPtr Device);

    /* MPS mode to release memory */
    [DllImport("htra_api.dll", EntryPoint = "MPS_DataBuffer_Free", CallingConvention = CallingConvention.Cdecl)]
    public static extern int MPS_DataBuffer_Free(ref IntPtr Device);

    /* -------------------------------------------The following functions are deprecated and should not be used anymore------------------------------------------------------ */

    /* The dev interface to control NX series devices to modify IP address (advanced application)*/
    [DllImport("htra_api.dll", EntryPoint = "Device_ChangeIPAddr", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_ChangeIPAddr(ref IntPtr Device, IPVersion_TypeDef ETH_IPVersion, uint8_t[] ETH_IPAddress, uint8_t[] SubnetMask);

    [DllImport("htra_api.dll", EntryPoint = "Device_GetIPAddr", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_GetIPAddr(uint8_t[][] ETH_IPAddress, uint8_t[][] SubnetMask, ref uint8_t IPAddressNum);

    /* The dev interface to control NX series devices to modify IP address (advanced application)*/
    [DllImport("htra_api.dll", EntryPoint = "Device_SetIPAddr", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_SetIPAddr(ref IntPtr Device, ref IPVersion_TypeDef ETH_IPVersion, uint8_t[] ETH_IPAddress, uint8_t[] SubnetMask);

    /* The dev interface to restart the network of NX series devices for the changed IP addresses to take effect (advanced application)*/
    [DllImport("htra_api.dll", EntryPoint = "Device_RestartNetwork", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_RestartNetwork(ref IntPtr Device);

    /* The dev interface to control the restart of NX series devices (advanced application)*/
    [DllImport("htra_api.dll", EntryPoint = "Device_Reboot", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_Reboot(ref IntPtr Device);

    /* The dev interface, update device MCU firmware (advanced application)*/
    [DllImport("htra_api.dll", EntryPoint = "Device_MCUFirmwareUpdate", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_MCUFirmwareUpdate(int DeviceNum, byte[] path);

    /* The dev interface, update device FPGA firmware (advanced application)*/
    [DllImport("htra_api.dll", EntryPoint = "Device_FPGAFirmwareUpdate", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_FPGAFirmwareUpdate(int DeviceNum, byte[] path);

    /*  SWP mode, reset spectrum data of MaxHold in SWP mode(advanced application) */
    [DllImport("htra_api.dll", EntryPoint = "SWP_MaxHoldReset", CallingConvention = CallingConvention.Cdecl)]
    public static extern void SWP_MaxHoldReset(ref IntPtr Device);

    /* IQS mode, Initiate timer - Externally triggered single synchronization*/
    [DllImport("htra_api.dll", EntryPoint = "IQS_SyncTimerByExtTrigger_Single", CallingConvention = CallingConvention.Cdecl)]
    public static extern int IQS_SyncTimerByExtTrigger_Single(ref IntPtr Device);

    /* DET mode, Initiate timer - Externally triggered single synchronization*/
    [DllImport("htra_api.dll", EntryPoint = "DET_SyncTimerByExtTrigger_Single", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DET_SyncTimerByExtTrigger_Single(ref IntPtr Device);

    /* RTA mode, Initiate timer - Externally triggered single synchronization*/
    [DllImport("htra_api.dll", EntryPoint = "RTA_SyncTimerByExtTrigger_Single", CallingConvention = CallingConvention.Cdecl)]
    public static extern int RTA_SyncTimerByExtTrigger_Single(ref IntPtr Device);

    /* -------------------------------------------The following are unnecessary function calls------------------------------------------------------ */

    /* The dev interface to print boot information, including UID,Model,HardwareVersion and etc.等 */
    [DllImport("htra_api.dll", EntryPoint = "Device_PrintBootInformation", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_PrintBootInformation(BootInfo_TypeDef BootInfo);

    /* The dev interface to get the current time(-H-M-S) and optionally print it or not */
    [DllImport("htra_api.dll", EntryPoint = "Device_GetLocalTime", CallingConvention = CallingConvention.Cdecl)]
    public static extern void Device_GetLocalTime(ref int hour, ref int min, ref int sec, int ifprint);

    /* SWP mode, Get debug information at a specified frequency in SWP mode (advanced applications) */
    [DllImport("htra_api.dll", EntryPoint = "SWP_GetDebugInfo", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SWP_GetDebugInfo(ref IntPtr Device, double Freq, ref SWP_DebugInfo_TypeDef DebugInfo);

    /* Cal mode, calibration function (CAL related interface)  */
    [DllImport("htra_api.dll", EntryPoint = "CAL_QDC_GetOptParam", CallingConvention = CallingConvention.Cdecl)]
    public static extern int CAL_QDC_GetOptParam(ref IQStream_TypeDef IQStream, ref CAL_QDCOptParam_TypeDef OptParam_O);

    // Mapping Query Device Startup Information Functions
    [DllImport("htra_api.dll", EntryPoint = "Device_QueryBootInfo", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_QueryBootInfo(int DeviceNum, ref BootProfile_TypeDef BootProfile, ref BootInfo_TypeDef BootInfo, ref uint UID_H32);

    // Mapping Check Device Availability Functions
    [DllImport("htra_api.dll", EntryPoint = "Device_CheckDeviceAvailability", CallingConvention = CallingConvention.Cdecl)]
    public static extern int Device_CheckDeviceAvailability(int DeviceNum, ref BootProfile_TypeDef BootProfile);

    /* ------------------------------------------- The following is signal generator control (SIG and VSG related interface)------------------------------------------------------ */


    /* SIG parameters are initialized */
    [DllImport("htra_api.dll", EntryPoint = "SIG_ProfileDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SIG_ProfileDeInit(ref IntPtr Device, ref SIG_Profile_TypeDef Profile);
    /* Configuration of signal generator working state */
    [DllImport("htra_api.dll", EntryPoint = "SIG_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SIG_Configuration(ref IntPtr Device, ref SIG_Profile_TypeDef ProfileIn, ref SIG_Profile_TypeDef ProfileOut, ref SIG_Info_TypeDef SIG_Info);
    /* The signal generator updates the packet header and packet tail sizes */
    [DllImport("htra_api.dll", EntryPoint = "SIG_UpdateHeaderTailSize", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SIG_UpdateHeaderTailSize(ref IntPtr Device, ref uint32_t HeaderSize, ref uint32_t TailSize);
    /* SIG sending IQStream */
    [DllImport("htra_api.dll", EntryPoint = "SIG_SendIQStream", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SIG_SendIQStream(ref IntPtr Device, byte[] DataBuffer, int DataSize);
    /* SIG bus trigger start */
    [DllImport("htra_api.dll", EntryPoint = "SIG_BusTriggerStart", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SIG_BusTriggerStart(ref IntPtr Device);
    /* SIG bus trigger stop */
    [DllImport("htra_api.dll", EntryPoint = "SIG_BusTriggerStop", CallingConvention = CallingConvention.Cdecl)]
    public static extern int SIG_BusTriggerStop(ref IntPtr Device);

    /* VSG parameters are initialized */
    [DllImport("htra_api.dll", EntryPoint = "VSG_ProfileDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VSG_ProfileDeInit(ref IntPtr Device, ref VSG_Profile_TypeDef Profile);
    /* Configuration of VSG working state */
    [DllImport("htra_api.dll", EntryPoint = "VSG_Configuration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VSG_Configuration(ref IntPtr Device, ref VSG_Profile_TypeDef ProfileIn, ref VSG_Profile_TypeDef ProfileOut, ref VSG_Info_TypeDef VSG_Info);
    /* The starting address of the VSG configuration download */
    [DllImport("htra_api.dll", EntryPoint = "VSG_Download_SetStartAddress", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VSG_Download_SetStartAddress(ref IntPtr Device, uint32_t StartAddress);
    /* VSG in IQStream file (single)*/
    [DllImport("htra_api.dll", EntryPoint = "VSG_Download_IQFile", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VSG_Download_IQFile(ref IntPtr Device, byte[] File, ref VSG_FileInfo_TypeDef VSG_FileInfo);

    /* The VSG updates the packet header and packet tail sizes */
    [DllImport("htra_api.dll", EntryPoint = "VSG_UpdateHeaderTailSize", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VSG_UpdateHeaderTailSize(ref IntPtr Device, ref uint32_t HeaderSize, ref uint32_t TailSize);
    /* The VSG downloads IQStream data */
    [DllImport("htra_api.dll", EntryPoint = "VSG_Download_IQData", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VSG_Download_IQData(ref IntPtr Device, ref int16_t I, ref int16_t Q, uint32_t length, ref uint32_t StartAddress, ref uint32_t StopAddress);
    /* The VSG downloads IQStream data(interleave)*/
    [DllImport("htra_api.dll", EntryPoint = "VSG_Download_IQData_Interleave", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VSG_Download_IQData_Interleave(ref IntPtr Device, ref int16_t IQ, uint32_t length, ref uint32_t StartAddress, ref uint32_t StopAddress);
    /* VSG bus trigger start*/
    [DllImport("htra_api.dll", EntryPoint = "VSG_BusTriggerStart", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VSG_BusTriggerStart(ref IntPtr Device);
    /* VSG bus trigger stop */
    [DllImport("htra_api.dll", EntryPoint = "VSG_PrivateParamDeInit", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VSG_PrivateParamDeInit(ref IntPtr Device, ref VSG_PrivateParam_TypeDef VSG_PrivateParam);
    /* VSG updates the private parameters */
    [DllImport("htra_api.dll", EntryPoint = "VSG_UpdatePrivateParam", CallingConvention = CallingConvention.Cdecl)]
    public static extern int VSG_UpdatePrivateParam(ref IntPtr Device, ref VSG_PrivateParam_TypeDef VSG_PrivateParam);

    /* Analog modulation to generate the signal waveform to be modulated */
    [DllImport("htra_api.dll", EntryPoint = "ASD_GenerateSignalWaveform", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ASD_GenerateSignalWaveform(ref IntPtr AnalogMod, AnalogSignalType_TypeDef AnalogSignalType, double SignalFrequency, double SampleRate, int resetflag, ref float m, ref int SamplePoints);

    /* The modulation is simulated to produce the final FM modulated waveform */
    [DllImport("htra_api.dll", EntryPoint = "ASD_GenerateFMWaveform", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ASD_GenerateFMWaveform(ref IntPtr AnalogMod, ref SIG_FM_Profile_TypeDef SIG_FM_Profile, ref float m, ref short I, ref short Q, ref int SamplePoints);
    [DllImport("htra_api.dll", EntryPoint = "ASD_GeneratePulseWaveform", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ASD_GeneratePulseWaveform(ref IntPtr AnalogMod, ref SIG_Pulse_Profile_TypeDef SIG_Pulse_Profile, ref short I, ref short Q, ref int SamplePoints);

    /* The modulation is simulated to produce the final AM modulated waveform */
    [DllImport("htra_api.dll", EntryPoint = "ASD_GenerateAMWaveform", CallingConvention = CallingConvention.Cdecl)]
    public static extern int ASD_GenerateAMWaveform(ref IntPtr AnalogMod, ref SIG_AM_Profile_TypeDef SIG_AM_Profile, ref float m, ref short I, ref short Q, ref int SamplePoints);

    /* DSP auxiliary function that performs IF calibration */
    [DllImport("htra_api.dll", EntryPoint = "DSP_IFCalibration", CallingConvention = CallingConvention.Cdecl)]
    public static extern int DSP_IFCalibration(ref IntPtr Device, ref double Frequency, ref float Spectrum, UInt32 SpectrumPoints);

    /* DSP auxiliary function to obtain the window type factor of the specified window type */
    [DllImport("htra_api.dll", EntryPoint = "DSP_GetWindowCoefficient", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_GetWindowCoefficient(Window_TypeDef Window, int n, double[] Coefficient);

    /* DSP auxiliary function, performs windowing*/
    [DllImport("htra_api.dll", EntryPoint = "DSP_Windowing", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_Windowing(DataFormat_TypeDef DataFormat, IntPtr Data_I, ref double Window, uint32_t Samples, ref double Data_O);

    /* DSP auxiliary function to obtain white gaussian noise */
    [DllImport("htra_api.dll", EntryPoint = "DSP_GetWhiteGaussianNoise", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_GetWhiteGaussianNoise(uint32_t n, ref float noise);

    /* DSP auxiliary function, convolution */
    [DllImport("htra_api.dll", EntryPoint = "DSP_Convolution", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_Convolution(float[] x, float[] y, int x_size, int y_size, float[] Output);

    /* DSP auxiliary function, decimate */
    [DllImport("htra_api.dll", EntryPoint = "DSP_Decimate", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_Decimate(float[] xn, float[] yn, uint64_t size, uint32_t DecimateFactor);

    /* DSP auxiliary function to generate a sine wave */
    [DllImport("htra_api.dll", EntryPoint = "DSP_GenerateSineWaveform", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_GenerateSineWaveform(float[] yn, ref Sin_TypeDef Sin_Profile);

    /* DSP auxiliary function, NCO is performed to generate sine and cosine signals */
    [DllImport("htra_api.dll", EntryPoint = "DSP_NCO_Execute", CallingConvention = CallingConvention.Cdecl)]
    public static extern void DSP_NCO_Execute(ref Sin_TypeDef NCO_Profile, ref float sin, ref float cos);

}
